"use strict";
const { createPlugin } = require('stylelint');
const atRuleParamIndex = require('stylelint/lib/utils/atRuleParamIndex');
const declarationValueIndex = require('stylelint/lib/utils/declarationValueIndex');
const getDimension = require('stylelint/lib/utils/getDimension');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const valueParser = require('postcss-value-parser');
const ruleName = 'stylistic/unit-case';
const messages = ruleMessages(ruleName, {
    expected: (actual, expected) => `Expected "${actual}" to be "${expected}"`,
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/unit-case',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
        actual: primary,
        possible: ['lower', 'upper'],
    });
    if (!validOptions) {
        return;
    }
    function check(node, checkedValue, getIndex) {
        const problems = [];
        function processValue(valueNode) {
            const { number, unit } = getDimension(valueNode);
            if (!number || !unit)
                return false;
            const expectedUnit = primary === 'lower' ? unit.toLowerCase() : unit.toUpperCase();
            if (unit === expectedUnit) {
                return false;
            }
            const index = getIndex(node);
            problems.push({
                index: index + valueNode.sourceIndex + number.length,
                endIndex: index + valueNode.sourceEndIndex,
                message: messages.expected(unit, expectedUnit),
            });
            return true;
        }
        const parsedValue = valueParser(checkedValue).walk((valueNode) => {
            let needFix = false;
            const { value } = valueNode;
            if (valueNode.type === 'function' && value.toLowerCase() === 'url') {
                return false;
            }
            if (value.includes('*')) {
                value.split('*').some((val) => processValue({
                    ...valueNode,
                    sourceIndex: value.indexOf(val) + val.length + 1,
                    value: val,
                }));
            }
            needFix = processValue(valueNode);
            if (needFix && context.fix) {
                valueNode.value = primary === 'lower' ? value.toLowerCase() : value.toUpperCase();
            }
        });
        if (problems.length) {
            if (context.fix) {
                if ('name' in node && node.name === 'media') {
                    node.params = parsedValue.toString();
                }
                else if ('value' in node) {
                    node.value = parsedValue.toString();
                }
            }
            else {
                for (const err of problems) {
                    report({
                        index: err.index,
                        endIndex: err.endIndex,
                        message: err.message,
                        node,
                        result,
                        ruleName,
                    });
                }
            }
        }
    }
    root.walkAtRules((atRule) => {
        if (!/^media$/i.test(atRule.name) && !('variable' in atRule)) {
            return;
        }
        check(atRule, atRule.params, atRuleParamIndex);
    });
    root.walkDecls((decl) => check(decl, decl.value, declarationValueIndex));
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
