"use strict";
const { createPlugin } = require('stylelint');
const valueParser = require('postcss-value-parser');
const atRuleParamIndex = require('stylelint/lib/utils/atRuleParamIndex');
const declarationValueIndex = require('stylelint/lib/utils/declarationValueIndex');
const isStandardSyntaxRule = require('stylelint/lib/utils/isStandardSyntaxRule');
const parseSelector = require('stylelint/lib/utils/parseSelector');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const { isBoolean, assertString } = require('stylelint/lib/utils/validateTypes');
const { isAtRule } = require('stylelint/lib/utils/typeGuards');
const ruleName = 'stylistic/string-quotes';
const messages = ruleMessages(ruleName, {
    expected: (q) => `Expected ${q} quotes`,
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/string-quotes',
    fixable: true,
};
const singleQuote = '\'';
const doubleQuote = '"';
const rule = (primary, secondaryOptions, context) => {
    const correctQuote = primary === 'single' ? singleQuote : doubleQuote;
    const erroneousQuote = primary === 'single' ? doubleQuote : singleQuote;
    return (root, result) => {
        const validOptions = validateOptions(result, ruleName, {
            actual: primary,
            possible: ['single', 'double'],
        }, {
            actual: secondaryOptions,
            possible: {
                avoidEscape: [isBoolean],
            },
            optional: true,
        });
        if (!validOptions) {
            return;
        }
        const avoidEscape = secondaryOptions && secondaryOptions.avoidEscape !== undefined
            ? secondaryOptions.avoidEscape
            : true;
        root.walk((node) => {
            switch (node.type) {
                case 'atrule':
                    checkDeclOrAtRule(node, node.params, atRuleParamIndex);
                    break;
                case 'decl':
                    checkDeclOrAtRule(node, node.value, declarationValueIndex);
                    break;
                case 'rule':
                    checkRule(node);
                    break;
            }
        });
        function checkRule(ruleNode) {
            if (!isStandardSyntaxRule(ruleNode)) {
                return;
            }
            if (!ruleNode.selector.includes('[') || !ruleNode.selector.includes('=')) {
                return;
            }
            const fixPositions = [];
            parseSelector(ruleNode.selector, result, ruleNode, (selectorTree) => {
                let selectorFixed = false;
                selectorTree.walkAttributes((attributeNode) => {
                    if (!attributeNode.quoted) {
                        return;
                    }
                    if (attributeNode.quoteMark === correctQuote && avoidEscape) {
                        assertString(attributeNode.value);
                        const needsCorrectEscape = attributeNode.value.includes(correctQuote);
                        const needsOtherEscape = attributeNode.value.includes(erroneousQuote);
                        if (needsOtherEscape) {
                            return;
                        }
                        if (needsCorrectEscape) {
                            if (context.fix) {
                                selectorFixed = true;
                                attributeNode.quoteMark = erroneousQuote;
                            }
                            else {
                                report({
                                    message: messages.expected(primary === 'single' ? 'double' : primary),
                                    node: ruleNode,
                                    index: attributeNode.sourceIndex + attributeNode.offsetOf('value'),
                                    result,
                                    ruleName,
                                });
                            }
                        }
                    }
                    if (attributeNode.quoteMark === erroneousQuote) {
                        if (avoidEscape) {
                            assertString(attributeNode.value);
                            const needsCorrectEscape = attributeNode.value.includes(correctQuote);
                            const needsOtherEscape = attributeNode.value.includes(erroneousQuote);
                            if (needsOtherEscape) {
                                if (context.fix) {
                                    selectorFixed = true;
                                    attributeNode.quoteMark = correctQuote;
                                }
                                else {
                                    report({
                                        message: messages.expected(primary),
                                        node: ruleNode,
                                        index: attributeNode.sourceIndex + attributeNode.offsetOf('value'),
                                        result,
                                        ruleName,
                                    });
                                }
                                return;
                            }
                            if (needsCorrectEscape) {
                                return;
                            }
                        }
                        if (context.fix) {
                            selectorFixed = true;
                            attributeNode.quoteMark = correctQuote;
                        }
                        else {
                            report({
                                message: messages.expected(primary),
                                node: ruleNode,
                                index: attributeNode.sourceIndex + attributeNode.offsetOf('value'),
                                result,
                                ruleName,
                            });
                        }
                    }
                });
                if (selectorFixed) {
                    ruleNode.selector = selectorTree.toString();
                }
            });
            for (const fixIndex of fixPositions) {
                ruleNode.selector = replaceQuote(ruleNode.selector, fixIndex, correctQuote);
            }
        }
        function checkDeclOrAtRule(node, value, getIndex) {
            const fixPositions = [];
            if (!value.includes(erroneousQuote)) {
                return;
            }
            if (isAtRule(node) && node.name === 'charset') {
                return;
            }
            valueParser(value).walk((valueNode) => {
                if (valueNode.type === 'string' && valueNode.quote === erroneousQuote) {
                    const needsEscape = valueNode.value.includes(correctQuote);
                    if (avoidEscape && needsEscape) {
                        return;
                    }
                    const openIndex = valueNode.sourceIndex;
                    if (context.fix && !needsEscape) {
                        const closeIndex = openIndex + valueNode.value.length + erroneousQuote.length;
                        fixPositions.push(openIndex, closeIndex);
                    }
                    else {
                        report({
                            message: messages.expected(primary),
                            node,
                            index: getIndex(node) + openIndex,
                            result,
                            ruleName,
                        });
                    }
                }
            });
            for (const fixIndex of fixPositions) {
                if (isAtRule(node)) {
                    node.params = replaceQuote(node.params, fixIndex, correctQuote);
                }
                else {
                    node.value = replaceQuote(node.value, fixIndex, correctQuote);
                }
            }
        }
    };
};
function replaceQuote(string, index, replace) {
    return string.substring(0, index) + replace + string.substring(index + replace.length);
}
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
