"use strict";
const { createPlugin } = require('stylelint');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const whitespaceChecker = require('stylelint/lib/utils/whitespaceChecker');
const selectorListCommaWhitespaceChecker = require('stylelint/lib/rules/selectorListCommaWhitespaceChecker');
const ruleName = 'stylistic/selector-list-comma-space-before';
const messages = ruleMessages(ruleName, {
    expectedBefore: () => 'Expected single space before ","',
    rejectedBefore: () => 'Unexpected whitespace before ","',
    expectedBeforeSingleLine: () => 'Expected single space before "," in a single-line list',
    rejectedBeforeSingleLine: () => 'Unexpected whitespace before "," in a single-line list',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/selector-list-comma-space-before',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => {
    const checker = whitespaceChecker('space', primary, messages);
    return (root, result) => {
        const validOptions = validateOptions(result, ruleName, {
            actual: primary,
            possible: ['always', 'never', 'always-single-line', 'never-single-line'],
        });
        if (!validOptions) {
            return;
        }
        let fixData;
        selectorListCommaWhitespaceChecker({
            root,
            result,
            locationChecker: checker.before,
            checkedRuleName: ruleName,
            fix: context.fix
                ? (ruleNode, index) => {
                    fixData = fixData || new Map();
                    const commaIndices = fixData.get(ruleNode) || [];
                    commaIndices.push(index);
                    fixData.set(ruleNode, commaIndices);
                    return true;
                }
                : null,
        });
        if (fixData) {
            for (const [ruleNode, commaIndices] of fixData.entries()) {
                let selector = ruleNode.raws.selector ? ruleNode.raws.selector.raw : ruleNode.selector;
                for (const index of commaIndices.sort((a, b) => b - a)) {
                    let beforeSelector = selector.slice(0, index);
                    const afterSelector = selector.slice(index);
                    if (primary.includes('always')) {
                        beforeSelector = beforeSelector.replace(/\s*$/, ' ');
                    }
                    else if (primary.includes('never')) {
                        beforeSelector = beforeSelector.replace(/\s*$/, '');
                    }
                    selector = beforeSelector + afterSelector;
                }
                if (ruleNode.raws.selector) {
                    ruleNode.raws.selector.raw = selector;
                }
                else {
                    ruleNode.selector = selector;
                }
            }
        }
    };
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
