"use strict";
const { createPlugin } = require('stylelint');
const styleSearch = require('style-search');
const isStandardSyntaxRule = require('stylelint/lib/utils/isStandardSyntaxRule');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const whitespaceChecker = require('stylelint/lib/utils/whitespaceChecker');
const ruleName = 'stylistic/selector-list-comma-newline-after';
const messages = ruleMessages(ruleName, {
    expectedAfter: () => 'Expected newline after ","',
    expectedAfterMultiLine: () => 'Expected newline after "," in a multi-line list',
    rejectedAfterMultiLine: () => 'Unexpected whitespace after "," in a multi-line list',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/selector-list-comma-newline-after',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => {
    const checker = whitespaceChecker('newline', primary, messages);
    return (root, result) => {
        const validOptions = validateOptions(result, ruleName, {
            actual: primary,
            possible: ['always', 'always-multi-line', 'never-multi-line'],
        });
        if (!validOptions) {
            return;
        }
        root.walkRules((ruleNode) => {
            if (!isStandardSyntaxRule(ruleNode)) {
                return;
            }
            const selector = ruleNode.raws.selector ? ruleNode.raws.selector.raw : ruleNode.selector;
            const fixIndices = [];
            styleSearch({
                source: selector,
                target: ',',
                functionArguments: 'skip',
            }, (match) => {
                const nextChars = selector.slice(match.endIndex);
                if (/^\s+\/\//.test(nextChars)) {
                    return;
                }
                const indextoCheckAfter = /^\s+\/\*/.test(nextChars)
                    ? selector.indexOf('*/', match.endIndex) + 1
                    : match.startIndex;
                checker.afterOneOnly({
                    source: selector,
                    index: indextoCheckAfter,
                    err: (m) => {
                        if (context.fix) {
                            fixIndices.push(indextoCheckAfter + 1);
                            return;
                        }
                        report({
                            message: m,
                            node: ruleNode,
                            index: match.startIndex,
                            result,
                            ruleName,
                        });
                    },
                });
            });
            if (fixIndices.length) {
                let fixedSelector = selector;
                for (const index of fixIndices.sort((a, b) => b - a)) {
                    const beforeSelector = fixedSelector.slice(0, index);
                    let afterSelector = fixedSelector.slice(index);
                    if (primary.startsWith('always')) {
                        afterSelector = context.newline + afterSelector;
                    }
                    else if (primary.startsWith('never-multi-line')) {
                        afterSelector = afterSelector.replace(/^\s*/, '');
                    }
                    fixedSelector = beforeSelector + afterSelector;
                }
                if (ruleNode.raws.selector) {
                    ruleNode.raws.selector.raw = fixedSelector;
                }
                else {
                    ruleNode.selector = fixedSelector;
                }
            }
        });
    };
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
