"use strict";
const { createPlugin } = require('stylelint');
const valueParser = require('postcss-value-parser');
const atRuleParamIndex = require('stylelint/lib/utils/atRuleParamIndex');
const declarationValueIndex = require('stylelint/lib/utils/declarationValueIndex');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const { isAtRule } = require('stylelint/lib/utils/typeGuards');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const ruleName = 'stylistic/number-no-trailing-zeros';
const messages = ruleMessages(ruleName, {
    rejected: 'Unexpected trailing zero(s)',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/number-no-trailing-zeros',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, { actual: primary });
    if (!validOptions) {
        return;
    }
    root.walkAtRules((atRule) => {
        if (atRule.name.toLowerCase() === 'import') {
            return;
        }
        check(atRule, atRule.params);
    });
    root.walkDecls((decl) => check(decl, decl.value));
    function check(node, value) {
        const fixPositions = [];
        if (!value.includes('.')) {
            return;
        }
        valueParser(value).walk((valueNode) => {
            if (valueNode.type === 'function' && valueNode.value.toLowerCase() === 'url') {
                return false;
            }
            if (valueNode.type !== 'word') {
                return;
            }
            const match = /\.(\d{0,100}?)(0+)(?:\D|$)/.exec(valueNode.value);
            if (match == null || match[1] == null || match[2] == null) {
                return;
            }
            const index = valueNode.sourceIndex + match.index + 1 + match[1].length;
            const startIndex = match[1].length > 0 ? index : index - 1;
            const endIndex = index + match[2].length;
            if (context.fix) {
                fixPositions.unshift({
                    startIndex,
                    endIndex,
                });
                return;
            }
            const baseIndex = isAtRule(node) ? atRuleParamIndex(node) : declarationValueIndex(node);
            report({
                message: messages.rejected,
                node,
                index: baseIndex + index,
                result,
                ruleName,
            });
        });
        if (fixPositions.length) {
            for (const fixPosition of fixPositions) {
                const { startIndex } = fixPosition;
                const { endIndex } = fixPosition;
                if (isAtRule(node)) {
                    node.params = removeTrailingZeros(node.params, startIndex, endIndex);
                }
                else {
                    node.value = removeTrailingZeros(node.value, startIndex, endIndex);
                }
            }
        }
    }
};
function removeTrailingZeros(input, startIndex, endIndex) {
    return input.slice(0, startIndex) + input.slice(endIndex);
}
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
