"use strict";
const { createPlugin } = require('stylelint');
const valueParser = require('postcss-value-parser');
const atRuleParamIndex = require('stylelint/lib/utils/atRuleParamIndex');
const declarationValueIndex = require('stylelint/lib/utils/declarationValueIndex');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const { isAtRule } = require('stylelint/lib/utils/typeGuards');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const ruleName = 'stylistic/number-leading-zero';
const messages = ruleMessages(ruleName, {
    expected: 'Expected a leading zero',
    rejected: 'Unexpected leading zero',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/number-leading-zero',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
        actual: primary,
        possible: ['always', 'never'],
    });
    if (!validOptions) {
        return;
    }
    root.walkAtRules((atRule) => {
        if (atRule.name.toLowerCase() === 'import') {
            return;
        }
        check(atRule, atRule.params);
    });
    root.walkDecls((decl) => check(decl, decl.value));
    function check(node, value) {
        const neverFixPositions = [];
        const alwaysFixPositions = [];
        if (!value.includes('.')) {
            return;
        }
        valueParser(value).walk((valueNode) => {
            if (valueNode.type === 'function' && valueNode.value.toLowerCase() === 'url') {
                return false;
            }
            if (valueNode.type !== 'word') {
                return;
            }
            if (primary === 'always') {
                const match = /(?:\D|^)(\.\d+)/.exec(valueNode.value);
                if (match == null || match[0] == null || match[1] == null) {
                    return;
                }
                const capturingGroupIndex = match[0].length - match[1].length;
                const index = valueNode.sourceIndex + match.index + capturingGroupIndex;
                if (context.fix) {
                    alwaysFixPositions.unshift({
                        index,
                    });
                    return;
                }
                const baseIndex = isAtRule(node) ? atRuleParamIndex(node) : declarationValueIndex(node);
                complain(messages.expected, node, baseIndex + index);
            }
            if (primary === 'never') {
                const match = /(?:\D|^)(0+)(\.\d+)/.exec(valueNode.value);
                if (match == null || match[0] == null || match[1] == null || match[2] == null) {
                    return;
                }
                const capturingGroupIndex = match[0].length - (match[1].length + match[2].length);
                const index = valueNode.sourceIndex + match.index + capturingGroupIndex;
                if (context.fix) {
                    neverFixPositions.unshift({
                        startIndex: index,
                        endIndex: index + match[1].length,
                    });
                    return;
                }
                const baseIndex = isAtRule(node) ? atRuleParamIndex(node) : declarationValueIndex(node);
                complain(messages.rejected, node, baseIndex + index);
            }
        });
        if (alwaysFixPositions.length) {
            for (const fixPosition of alwaysFixPositions) {
                const { index } = fixPosition;
                if (isAtRule(node)) {
                    node.params = addLeadingZero(node.params, index);
                }
                else {
                    node.value = addLeadingZero(node.value, index);
                }
            }
        }
        if (neverFixPositions.length) {
            for (const fixPosition of neverFixPositions) {
                const { startIndex } = fixPosition;
                const { endIndex } = fixPosition;
                if (isAtRule(node)) {
                    node.params = removeLeadingZeros(node.params, startIndex, endIndex);
                }
                else {
                    node.value = removeLeadingZeros(node.value, startIndex, endIndex);
                }
            }
        }
    }
    function complain(message, node, index) {
        report({
            result,
            ruleName,
            message,
            node,
            index,
        });
    }
};
function addLeadingZero(input, index) {
    return input.slice(0, index) + '0' + input.slice(index);
}
function removeLeadingZeros(input, startIndex, endIndex) {
    return input.slice(0, startIndex) + input.slice(endIndex);
}
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
