"use strict";
const { createPlugin } = require('stylelint');
const styleSearch = require('style-search');
const isStandardSyntaxAtRule = require('stylelint/lib/utils/isStandardSyntaxAtRule');
const isStandardSyntaxRule = require('stylelint/lib/utils/isStandardSyntaxRule');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const { isAtRule } = require('stylelint/lib/utils/typeGuards');
const ruleName = 'stylistic/no-extra-semicolons';
const messages = ruleMessages(ruleName, {
    rejected: 'Unexpected extra semicolon',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/no-extra-semicolons',
    fixable: true,
};
function getOffsetByNode(node) {
    if (node.parent && node.parent.document) {
        return 0;
    }
    const root = node.root();
    if (!root.source)
        throw new Error('The root node must have a source');
    if (!node.source)
        throw new Error('The node must have a source');
    if (!node.source.start)
        throw new Error('The source must have a start position');
    const string = root.source.input.css;
    const nodeColumn = node.source.start.column;
    const nodeLine = node.source.start.line;
    let line = 1;
    let column = 1;
    let index = 0;
    for (let i = 0; i < string.length; i += 1) {
        if (column === nodeColumn && nodeLine === line) {
            index = i;
            break;
        }
        if (string[i] === '\n') {
            column = 1;
            line += 1;
        }
        else {
            column += 1;
        }
    }
    return index;
}
const rule = (primary, _secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, { actual: primary });
    if (!validOptions) {
        return;
    }
    if (root.raws.after && root.raws.after.trim().length !== 0) {
        const rawAfterRoot = root.raws.after;
        const fixSemiIndices = [];
        styleSearch({ source: rawAfterRoot, target: ';' }, (match) => {
            if (context.fix) {
                fixSemiIndices.push(match.startIndex);
                return;
            }
            if (!root.source)
                throw new Error('The root node must have a source');
            complain(root.source.input.css.length - rawAfterRoot.length + match.startIndex);
        });
        if (fixSemiIndices.length) {
            root.raws.after = removeIndices(rawAfterRoot, fixSemiIndices);
        }
    }
    root.walk((node) => {
        if (isAtRule(node) && !isStandardSyntaxAtRule(node)) {
            return;
        }
        if (node.type === 'rule' && !isStandardSyntaxRule(node)) {
            return;
        }
        if (node.raws.before && node.raws.before.trim().length !== 0) {
            const rawBeforeNode = node.raws.before;
            const allowedSemi = 0;
            const rawBeforeIndexStart = 0;
            const fixSemiIndices = [];
            styleSearch({ source: rawBeforeNode, target: ';' }, (match, count) => {
                if (count === allowedSemi) {
                    return;
                }
                if (context.fix) {
                    fixSemiIndices.push(match.startIndex - rawBeforeIndexStart);
                    return;
                }
                complain(getOffsetByNode(node) - rawBeforeNode.length + match.startIndex);
            });
            if (fixSemiIndices.length) {
                node.raws.before = removeIndices(rawBeforeNode, fixSemiIndices);
            }
        }
        if (typeof node.raws.after === 'string' && node.raws.after.trim().length !== 0) {
            const rawAfterNode = node.raws.after;
            if ('last' in node
                && node.last
                && node.last.type === 'atrule'
                && !isStandardSyntaxAtRule(node.last)) {
                return;
            }
            const fixSemiIndices = [];
            styleSearch({ source: rawAfterNode, target: ';' }, (match) => {
                if (context.fix) {
                    fixSemiIndices.push(match.startIndex);
                    return;
                }
                const index = getOffsetByNode(node)
                    + node.toString().length
                    - 1
                    - rawAfterNode.length
                    + match.startIndex;
                complain(index);
            });
            if (fixSemiIndices.length) {
                node.raws.after = removeIndices(rawAfterNode, fixSemiIndices);
            }
        }
        if (typeof node.raws.ownSemicolon === 'string') {
            const rawOwnSemicolon = node.raws.ownSemicolon;
            const allowedSemi = 0;
            const fixSemiIndices = [];
            styleSearch({ source: rawOwnSemicolon, target: ';' }, (match, count) => {
                if (count === allowedSemi) {
                    return;
                }
                if (context.fix) {
                    fixSemiIndices.push(match.startIndex);
                    return;
                }
                const index = getOffsetByNode(node)
                    + node.toString().length
                    - rawOwnSemicolon.length
                    + match.startIndex;
                complain(index);
            });
            if (fixSemiIndices.length) {
                node.raws.ownSemicolon = removeIndices(rawOwnSemicolon, fixSemiIndices);
            }
        }
    });
    function complain(index) {
        report({
            message: messages.rejected,
            node: root,
            index,
            result,
            ruleName,
        });
    }
    function removeIndices(str, indices) {
        for (const index of indices.reverse()) {
            str = str.slice(0, index) + str.slice(index + 1);
        }
        return str;
    }
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
