"use strict";
const { createPlugin } = require('stylelint');
const styleSearch = require('style-search');
const isOnlyWhitespace = require('stylelint/lib/utils/isOnlyWhitespace');
const isStandardSyntaxComment = require('stylelint/lib/utils/isStandardSyntaxComment');
const optionsMatches = require('stylelint/lib/utils/optionsMatches');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const { isAtRule, isComment, isDeclaration, isRule } = require('stylelint/lib/utils/typeGuards');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const ruleName = 'stylistic/no-eol-whitespace';
const messages = ruleMessages(ruleName, {
    rejected: 'Unexpected whitespace at end of line',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/no-eol-whitespace',
    fixable: true,
};
const whitespacesToReject = new Set([' ', '\t']);
function fixString(str) {
    return str.replace(/[ \t]+$/, '');
}
function findErrorStartIndex(lastEOLIndex, string, { ignoreEmptyLines, isRootFirst }) {
    const eolWhitespaceIndex = lastEOLIndex - 1;
    if (!whitespacesToReject.has(string.charAt(eolWhitespaceIndex))) {
        return -1;
    }
    if (ignoreEmptyLines) {
        const beforeNewlineIndex = string.lastIndexOf('\n', eolWhitespaceIndex);
        if (beforeNewlineIndex >= 0 || isRootFirst) {
            const line = string.substring(beforeNewlineIndex, eolWhitespaceIndex);
            if (isOnlyWhitespace(line)) {
                return -1;
            }
        }
    }
    return eolWhitespaceIndex;
}
const rule = (primary, secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
        actual: primary,
    }, {
        optional: true,
        actual: secondaryOptions,
        possible: {
            ignore: ['empty-lines'],
        },
    });
    if (!validOptions) {
        return;
    }
    const ignoreEmptyLines = optionsMatches(secondaryOptions, 'ignore', 'empty-lines');
    if (context.fix) {
        fix(root);
    }
    const rootString = context.fix ? root.toString() : (root.source && root.source.input.css) || '';
    const reportFromIndex = (index) => {
        report({
            message: messages.rejected,
            node: root,
            index,
            result,
            ruleName,
        });
    };
    eachEolWhitespace(rootString, reportFromIndex, true);
    const errorIndex = findErrorStartIndex(rootString.length, rootString, {
        ignoreEmptyLines,
        isRootFirst: true,
    });
    if (errorIndex > -1) {
        reportFromIndex(errorIndex);
    }
    function eachEolWhitespace(string, callback, isRootFirst) {
        styleSearch({
            source: string,
            target: ['\n', '\r'],
            comments: 'check',
        }, (match) => {
            const index = findErrorStartIndex(match.startIndex, string, {
                ignoreEmptyLines,
                isRootFirst,
            });
            if (index > -1) {
                callback(index);
            }
        });
    }
    function fix(rootNode) {
        let isRootFirst = true;
        rootNode.walk((node) => {
            fixText(node.raws.before, (fixed) => {
                node.raws.before = fixed;
            }, isRootFirst);
            isRootFirst = false;
            if (isAtRule(node)) {
                fixText(node.raws.afterName, (fixed) => {
                    node.raws.afterName = fixed;
                });
                const rawsParams = node.raws.params;
                if (rawsParams) {
                    fixText(rawsParams.raw, (fixed) => {
                        rawsParams.raw = fixed;
                    });
                }
                else {
                    fixText(node.params, (fixed) => {
                        node.params = fixed;
                    });
                }
            }
            if (isRule(node)) {
                const rawsSelector = node.raws.selector;
                if (rawsSelector) {
                    fixText(rawsSelector.raw, (fixed) => {
                        rawsSelector.raw = fixed;
                    });
                }
                else {
                    fixText(node.selector, (fixed) => {
                        node.selector = fixed;
                    });
                }
            }
            if (isAtRule(node) || isRule(node) || isDeclaration(node)) {
                fixText(node.raws.between, (fixed) => {
                    node.raws.between = fixed;
                });
            }
            if (isDeclaration(node)) {
                const rawsValue = node.raws.value;
                if (rawsValue) {
                    fixText(rawsValue.raw, (fixed) => {
                        rawsValue.raw = fixed;
                    });
                }
                else {
                    fixText(node.value, (fixed) => {
                        node.value = fixed;
                    });
                }
            }
            if (isComment(node)) {
                fixText(node.raws.left, (fixed) => {
                    node.raws.left = fixed;
                });
                if (!isStandardSyntaxComment(node)) {
                    node.raws.right = node.raws.right && fixString(node.raws.right);
                }
                else {
                    fixText(node.raws.right, (fixed) => {
                        node.raws.right = fixed;
                    });
                }
                fixText(node.text, (fixed) => {
                    node.text = fixed;
                });
            }
            if (isAtRule(node) || isRule(node)) {
                fixText(node.raws.after, (fixed) => {
                    node.raws.after = fixed;
                });
            }
        });
        fixText(rootNode.raws.after, (fixed) => {
            rootNode.raws.after = fixed;
        }, isRootFirst);
        if (typeof rootNode.raws.after === 'string') {
            const lastEOL = Math.max(rootNode.raws.after.lastIndexOf('\n'), rootNode.raws.after.lastIndexOf('\r'));
            if (lastEOL !== rootNode.raws.after.length - 1) {
                rootNode.raws.after = rootNode.raws.after.slice(0, lastEOL + 1)
                    + fixString(rootNode.raws.after.slice(lastEOL + 1));
            }
        }
    }
    function fixText(value, fixFn, isRootFirst = false) {
        if (!value) {
            return;
        }
        let fixed = '';
        let lastIndex = 0;
        eachEolWhitespace(value, (index) => {
            const newlineIndex = index + 1;
            fixed += fixString(value.slice(lastIndex, newlineIndex));
            lastIndex = newlineIndex;
        }, isRootFirst);
        if (lastIndex) {
            fixed += value.slice(lastIndex);
            fixFn(fixed);
        }
    }
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
