"use strict";
const { createPlugin } = require('stylelint');
const valueParser = require('postcss-value-parser');
const atRuleParamIndex = require('stylelint/lib/utils/atRuleParamIndex');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const ruleName = 'stylistic/media-feature-parentheses-space-inside';
const messages = ruleMessages(ruleName, {
    expectedOpening: 'Expected single space after "("',
    rejectedOpening: 'Unexpected whitespace after "("',
    expectedClosing: 'Expected single space before ")"',
    rejectedClosing: 'Unexpected whitespace before ")"',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/media-feature-parentheses-space-inside',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
        actual: primary,
        possible: ['always', 'never'],
    });
    if (!validOptions) {
        return;
    }
    root.walkAtRules(/^media$/i, (atRule) => {
        const params = (atRule.raws.params && atRule.raws.params.raw) || atRule.params;
        const indexBoost = atRuleParamIndex(atRule);
        const problems = [];
        const parsedParams = valueParser(params).walk((node) => {
            if (node.type === 'function') {
                const len = valueParser.stringify(node).length;
                if (primary === 'never') {
                    if (/[ \t]/.test(node.before)) {
                        if (context.fix)
                            node.before = '';
                        problems.push({
                            message: messages.rejectedOpening,
                            index: node.sourceIndex + 1 + indexBoost,
                        });
                    }
                    if (/[ \t]/.test(node.after)) {
                        if (context.fix)
                            node.after = '';
                        problems.push({
                            message: messages.rejectedClosing,
                            index: node.sourceIndex - 2 + len + indexBoost,
                        });
                    }
                }
                else if (primary === 'always') {
                    if (node.before === '') {
                        if (context.fix)
                            node.before = ' ';
                        problems.push({
                            message: messages.expectedOpening,
                            index: node.sourceIndex + 1 + indexBoost,
                        });
                    }
                    if (node.after === '') {
                        if (context.fix)
                            node.after = ' ';
                        problems.push({
                            message: messages.expectedClosing,
                            index: node.sourceIndex - 2 + len + indexBoost,
                        });
                    }
                }
            }
        });
        if (problems.length) {
            if (context.fix) {
                atRule.params = parsedParams.toString();
                return;
            }
            for (const err of problems) {
                report({
                    message: err.message,
                    node: atRule,
                    index: err.index,
                    result,
                    ruleName,
                });
            }
        }
    });
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
