"use strict";
const { createPlugin } = require('stylelint');
const styleSearch = require('style-search');
const optionsMatches = require('stylelint/lib/utils/optionsMatches');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const { isNumber, isRegExp, isString, assert } = require('stylelint/lib/utils/validateTypes');
const ruleName = 'stylistic/max-line-length';
const messages = ruleMessages(ruleName, {
    expected: (max) => `Expected line length to be no more than ${max} ${max === 1 ? 'character' : 'characters'}`,
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/max-line-length',
};
const rule = (primary, secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
        actual: primary,
        possible: isNumber,
    }, {
        actual: secondaryOptions,
        possible: {
            ignore: ['non-comments', 'comments'],
            ignorePattern: [isString, isRegExp],
        },
        optional: true,
    });
    if (!validOptions) {
        return;
    }
    if (root.source == null) {
        throw new Error('The root node must have a source');
    }
    const EXCLUDED_PATTERNS = [
        /url\(\s*(\S.*\S)\s*\)/gi,
        /@import\s+(['"].*['"])/gi,
    ];
    const ignoreNonComments = optionsMatches(secondaryOptions, 'ignore', 'non-comments');
    const ignoreComments = optionsMatches(secondaryOptions, 'ignore', 'comments');
    const rootString = context.fix ? root.toString() : root.source.input.css;
    let skippedSubStrings = [];
    let skippedSubStringsIndex = 0;
    for (const pattern of EXCLUDED_PATTERNS) {
        for (const match of rootString.matchAll(pattern)) {
            const subMatch = match[1] || '';
            const startOfSubString = (match.index || 0) + (match[0] || '').indexOf(subMatch);
            skippedSubStrings.push([startOfSubString, startOfSubString + subMatch.length]);
        }
    }
    skippedSubStrings = skippedSubStrings.sort((a, b) => a[0] - b[0]);
    checkNewline({ endIndex: 0 });
    styleSearch({ source: rootString, target: ['\n'], comments: 'check' }, (match) => checkNewline(match));
    function complain(index) {
        report({
            index,
            result,
            ruleName,
            message: messages.expected(primary),
            node: root,
        });
    }
    function tryToPopSubString(start, end) {
        const skippedSubString = skippedSubStrings[skippedSubStringsIndex];
        assert(skippedSubString);
        const [startSubString, endSubString] = skippedSubString;
        if (end < startSubString) {
            return 0;
        }
        const excluded = Math.min(end, endSubString) - Math.max(start, startSubString);
        if (endSubString <= end) {
            skippedSubStringsIndex += 1;
        }
        return excluded;
    }
    function checkNewline(match) {
        let nextNewlineIndex = rootString.indexOf('\n', match.endIndex);
        if (rootString[nextNewlineIndex - 1] === '\r') {
            nextNewlineIndex -= 1;
        }
        if (nextNewlineIndex === -1) {
            nextNewlineIndex = rootString.length;
        }
        const rawLineLength = nextNewlineIndex - match.endIndex;
        const excludedLength = skippedSubStrings[skippedSubStringsIndex]
            ? tryToPopSubString(match.endIndex, nextNewlineIndex)
            : 0;
        const lineText = rootString.slice(match.endIndex, nextNewlineIndex);
        if (optionsMatches(secondaryOptions, 'ignorePattern', lineText)) {
            return;
        }
        if (rawLineLength - excludedLength <= primary) {
            return;
        }
        const complaintIndex = nextNewlineIndex - 1;
        if (ignoreComments) {
            if ('insideComment' in match && match.insideComment) {
                return;
            }
            const nextTwoChars = rootString.slice(match.endIndex).trim().slice(0, 2);
            if (nextTwoChars === '/*' || nextTwoChars === '//') {
                return;
            }
        }
        if (ignoreNonComments) {
            if ('insideComment' in match && match.insideComment) {
                return complain(complaintIndex);
            }
            const nextTwoChars = rootString.slice(match.endIndex).trim().slice(0, 2);
            if (nextTwoChars !== '/*' && nextTwoChars !== '//') {
                return;
            }
            return complain(complaintIndex);
        }
        const lineString = rootString.slice(match.endIndex, nextNewlineIndex);
        if (!lineString.replace(/^\s+/, '').includes(' ')) {
            return;
        }
        return complain(complaintIndex);
    }
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
