"use strict";
const { createPlugin } = require('stylelint');
const styleSearch = require('style-search');
const optionsMatches = require('stylelint/lib/utils/optionsMatches');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const { isNumber } = require('stylelint/lib/utils/validateTypes');
const ruleName = 'stylistic/max-empty-lines';
const messages = ruleMessages(ruleName, {
    expected: (max) => `Expected no more than ${max} empty ${max === 1 ? 'line' : 'lines'}`,
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/max-empty-lines',
    fixable: true,
};
const rule = (primary, secondaryOptions, context) => {
    let emptyLines = 0;
    let lastIndex = -1;
    return (root, result) => {
        const validOptions = validateOptions(result, ruleName, {
            actual: primary,
            possible: isNumber,
        }, {
            actual: secondaryOptions,
            possible: {
                ignore: ['comments'],
            },
            optional: true,
        });
        if (!validOptions) {
            return;
        }
        const ignoreComments = optionsMatches(secondaryOptions, 'ignore', 'comments');
        const getChars = replaceEmptyLines.bind(null, primary);
        if (context.fix) {
            root.walk((node) => {
                if (node.type === 'comment' && !ignoreComments) {
                    node.raws.left = getChars(node.raws.left);
                    node.raws.right = getChars(node.raws.right);
                }
                if (node.raws.before) {
                    node.raws.before = getChars(node.raws.before);
                }
            });
            const firstNodeRawsBefore = root.first && root.first.raws.before;
            const rootRawsAfter = root.raws.after;
            if ((root.document && root.document.constructor.name) !== 'Document') {
                if (firstNodeRawsBefore) {
                    root.first.raws.before = getChars(firstNodeRawsBefore, true);
                }
                if (rootRawsAfter) {
                    root.raws.after = replaceEmptyLines(primary === 0 ? 1 : primary, rootRawsAfter, true);
                }
            }
            else if (rootRawsAfter) {
                root.raws.after = replaceEmptyLines(primary === 0 ? 1 : primary, rootRawsAfter);
            }
            return;
        }
        emptyLines = 0;
        lastIndex = -1;
        const rootString = root.toString();
        styleSearch({
            source: rootString,
            target: /\r\n/.test(rootString) ? '\r\n' : '\n',
            comments: ignoreComments ? 'skip' : 'check',
        }, (match) => {
            checkMatch(rootString, match.startIndex, match.endIndex, root);
        });
        function checkMatch(source, matchStartIndex, matchEndIndex, node) {
            const eof = matchEndIndex === source.length;
            let problem = false;
            if (!matchStartIndex || lastIndex === matchStartIndex) {
                emptyLines += 1;
            }
            else {
                emptyLines = 0;
            }
            lastIndex = matchEndIndex;
            if (emptyLines > primary)
                problem = true;
            if (!eof && !problem)
                return;
            if (problem) {
                report({
                    message: messages.expected(primary),
                    node,
                    index: matchStartIndex,
                    result,
                    ruleName,
                });
            }
            if (eof && primary) {
                emptyLines += 1;
                if (emptyLines > primary && isEofNode(result.root, node)) {
                    report({
                        message: messages.expected(primary),
                        node,
                        index: matchEndIndex,
                        result,
                        ruleName,
                    });
                }
            }
        }
        function replaceEmptyLines(maxLines, str, isSpecialCase = false) {
            const repeatTimes = isSpecialCase ? maxLines : maxLines + 1;
            if (repeatTimes === 0 || typeof str !== 'string') {
                return '';
            }
            const emptyLFLines = '\n'.repeat(repeatTimes);
            const emptyCRLFLines = '\r\n'.repeat(repeatTimes);
            return /(?:\r\n)+/.test(str)
                ? str.replace(/(\r\n)+/g, ($1) => {
                    if ($1.length / 2 > repeatTimes) {
                        return emptyCRLFLines;
                    }
                    return $1;
                })
                : str.replace(/(\n)+/g, ($1) => {
                    if ($1.length > repeatTimes) {
                        return emptyLFLines;
                    }
                    return $1;
                });
        }
    };
};
function isEofNode(document, root) {
    if (!document || document.constructor.name !== 'Document' || !('type' in document)) {
        return true;
    }
    let after;
    if (root === document.last) {
        after = document.raws && document.raws.codeAfter;
    }
    else {
        const rootIndex = document.index(root);
        const nextNode = document.nodes[rootIndex + 1];
        after = nextNode && nextNode.raws && nextNode.raws.codeBefore;
    }
    return !String(after).trim();
}
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
