"use strict";
const { createPlugin } = require('stylelint');
const postcss = require('postcss');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const ruleName = 'stylistic/linebreaks';
const messages = ruleMessages(ruleName, {
    expected: (linebreak) => `Expected linebreak to be ${linebreak}`,
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/linebreaks',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
        actual: primary,
        possible: ['unix', 'windows'],
    });
    if (!validOptions) {
        return;
    }
    const shouldHaveCR = primary === 'windows';
    if (context.fix) {
        root.walk((node) => {
            if ('selector' in node) {
                node.selector = fixData(node.selector);
            }
            if ('value' in node) {
                node.value = fixData(node.value);
            }
            if ('text' in node) {
                node.text = fixData(node.text);
            }
            if (node.raws.before) {
                node.raws.before = fixData(node.raws.before);
            }
            if (typeof node.raws.after === 'string') {
                node.raws.after = fixData(node.raws.after);
            }
        });
        if (typeof root.raws.after === 'string') {
            root.raws.after = fixData(root.raws.after);
        }
    }
    else {
        if (root.source == null)
            throw new Error('The root node must have a source');
        const lines = root.source.input.css.split('\n');
        for (let [i, line] of lines.entries()) {
            if (i < lines.length - 1 && !line.includes('\r')) {
                line += '\n';
            }
            if (hasError(line)) {
                const lineNum = i + 1;
                const colNum = line.length;
                reportNewlineError(lineNum, colNum);
            }
        }
    }
    function hasError(dataToCheck) {
        const hasNewlineToVerify = /[\r\n]/.test(dataToCheck);
        const hasCR = hasNewlineToVerify ? /\r/.test(dataToCheck) : false;
        return hasNewlineToVerify && hasCR !== shouldHaveCR;
    }
    function fixData(data) {
        if (data) {
            let res = data.replace(/\r/g, '');
            if (shouldHaveCR) {
                res = res.replace(/\n/g, '\r\n');
            }
            return res;
        }
        return data;
    }
    function reportNewlineError(line, column) {
        const node = postcss.rule({
            source: {
                start: { line, column, offset: 0 },
                input: new postcss.Input(''),
            },
        });
        report({
            message: messages.expected(primary),
            node,
            result,
            ruleName,
        });
    }
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
