"use strict";
const { createPlugin } = require('stylelint');
const beforeBlockString = require('stylelint/lib/utils/beforeBlockString');
const hasBlock = require('stylelint/lib/utils/hasBlock');
const optionsMatches = require('stylelint/lib/utils/optionsMatches');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const styleSearch = require('style-search');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const { isAtRule, isDeclaration, isRoot, isRule } = require('stylelint/lib/utils/typeGuards');
const { isBoolean, isNumber, isString, assertString } = require('stylelint/lib/utils/validateTypes');
const ruleName = 'stylistic/indentation';
const messages = ruleMessages(ruleName, {
    expected: (x) => `Expected indentation of ${x}`,
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/indentation',
    fixable: true,
};
const rule = (primary, secondaryOptions = {}, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
        actual: primary,
        possible: [isNumber, 'tab'],
    }, {
        actual: secondaryOptions,
        possible: {
            baseIndentLevel: [isNumber, 'auto'],
            except: ['block', 'value', 'param'],
            ignore: ['value', 'param', 'inside-parens'],
            indentInsideParens: ['twice', 'once-at-root-twice-in-block'],
            indentClosingBrace: [isBoolean],
        },
        optional: true,
    });
    if (!validOptions) {
        return;
    }
    const spaceCount = isNumber(primary) ? primary : null;
    const indentChar = spaceCount == null ? '\t' : ' '.repeat(spaceCount);
    const warningWord = primary === 'tab' ? 'tab' : 'space';
    const { baseIndentLevel } = secondaryOptions;
    const { indentClosingBrace } = secondaryOptions;
    const legibleExpectation = (level) => {
        const count = spaceCount == null ? level : level * spaceCount;
        const quantifiedWarningWord = count === 1 ? warningWord : `${warningWord}s`;
        return `${count} ${quantifiedWarningWord}`;
    };
    root.walk((node) => {
        if (isRoot(node)) {
            return;
        }
        const nodeLevel = indentationLevel(node);
        const before = (node.raws.before || '').replace(/[*_]$/, '');
        const after = typeof node.raws.after === 'string' ? node.raws.after : '';
        const { parent } = node;
        if (!parent)
            throw new Error('A parent node must be present');
        const expectedOpeningBraceIndentation = indentChar.repeat(nodeLevel);
        const isFirstChild = parent.type === 'root' && parent.first === node;
        const lastIndexOfNewline = before.lastIndexOf('\n');
        if ((lastIndexOfNewline !== -1
            || (isFirstChild
                && (!getDocument(parent)
                    || (parent.raws.codeBefore && parent.raws.codeBefore.endsWith('\n')))))
            && before.slice(lastIndexOfNewline + 1) !== expectedOpeningBraceIndentation) {
            if (context.fix) {
                if (isFirstChild && isString(node.raws.before)) {
                    node.raws.before = node.raws.before.replace(/^[ \t]*(?=\S|$)/, expectedOpeningBraceIndentation);
                }
                node.raws.before = fixIndentation(node.raws.before, expectedOpeningBraceIndentation);
            }
            else {
                report({
                    message: messages.expected(legibleExpectation(nodeLevel)),
                    node,
                    result,
                    ruleName,
                });
            }
        }
        const closingBraceLevel = indentClosingBrace ? nodeLevel + 1 : nodeLevel;
        const expectedClosingBraceIndentation = indentChar.repeat(closingBraceLevel);
        if ((isRule(node) || isAtRule(node))
            && hasBlock(node)
            && after
            && after.includes('\n')
            && after.slice(after.lastIndexOf('\n') + 1) !== expectedClosingBraceIndentation) {
            if (context.fix) {
                node.raws.after = fixIndentation(node.raws.after, expectedClosingBraceIndentation);
            }
            else {
                report({
                    message: messages.expected(legibleExpectation(closingBraceLevel)),
                    node,
                    index: node.toString().length - 1,
                    result,
                    ruleName,
                });
            }
        }
        if (isDeclaration(node)) {
            checkValue(node, nodeLevel);
        }
        if (isRule(node)) {
            checkSelector(node, nodeLevel);
        }
        if (isAtRule(node)) {
            checkAtRuleParams(node, nodeLevel);
        }
    });
    function indentationLevel(node, level = 0) {
        if (!node.parent)
            throw new Error('A parent node must be present');
        if (isRoot(node.parent)) {
            return level + getRootBaseIndentLevel(node.parent, baseIndentLevel, primary);
        }
        let calculatedLevel;
        calculatedLevel = indentationLevel(node.parent, level + 1);
        if (optionsMatches(secondaryOptions, 'except', 'block')
            && (isRule(node) || isAtRule(node))
            && hasBlock(node)) {
            calculatedLevel -= 1;
        }
        return calculatedLevel;
    }
    function checkValue(decl, declLevel) {
        if (!decl.value.includes('\n')) {
            return;
        }
        if (optionsMatches(secondaryOptions, 'ignore', 'value')) {
            return;
        }
        const declString = decl.toString();
        const valueLevel = optionsMatches(secondaryOptions, 'except', 'value')
            ? declLevel
            : declLevel + 1;
        checkMultilineBit(declString, valueLevel, decl);
    }
    function checkSelector(ruleNode, ruleLevel) {
        const { selector } = ruleNode;
        if (ruleNode.params) {
            ruleLevel += 1;
        }
        checkMultilineBit(selector, ruleLevel, ruleNode);
    }
    function checkAtRuleParams(atRule, ruleLevel) {
        if (optionsMatches(secondaryOptions, 'ignore', 'param')) {
            return;
        }
        const paramLevel = optionsMatches(secondaryOptions, 'except', 'param')
            || atRule.name === 'nest'
            || atRule.name === 'at-root'
            ? ruleLevel
            : ruleLevel + 1;
        checkMultilineBit(beforeBlockString(atRule).trim(), paramLevel, atRule);
    }
    function checkMultilineBit(source, newlineIndentLevel, node) {
        if (!source.includes('\n')) {
            return;
        }
        const fixPositions = [];
        let parentheticalDepth = 0;
        const ignoreInsideParans = optionsMatches(secondaryOptions, 'ignore', 'inside-parens');
        styleSearch({
            source,
            target: '\n',
            outsideParens: ignoreInsideParans,
        }, (match, matchCount) => {
            const precedesClosingParenthesis = /^[ \t]*\)/.test(source.slice(match.startIndex + 1));
            if (ignoreInsideParans && (precedesClosingParenthesis || match.insideParens)) {
                return;
            }
            let expectedIndentLevel = newlineIndentLevel;
            if (!ignoreInsideParans && match.insideParens) {
                if (matchCount === 1)
                    parentheticalDepth -= 1;
                let newlineIndex = match.startIndex;
                if (source[match.startIndex - 1] === '\r') {
                    newlineIndex -= 1;
                }
                const followsOpeningParenthesis = /\([ \t]*$/.test(source.slice(0, newlineIndex));
                if (followsOpeningParenthesis) {
                    parentheticalDepth += 1;
                }
                const followsOpeningBrace = /\{[ \t]*$/.test(source.slice(0, newlineIndex));
                if (followsOpeningBrace) {
                    parentheticalDepth += 1;
                }
                const startingClosingBrace = /^[ \t]*\}/.test(source.slice(match.startIndex + 1));
                if (startingClosingBrace) {
                    parentheticalDepth -= 1;
                }
                expectedIndentLevel += parentheticalDepth;
                if (precedesClosingParenthesis) {
                    parentheticalDepth -= 1;
                }
                switch (secondaryOptions.indentInsideParens) {
                    case 'twice':
                        if (!precedesClosingParenthesis || indentClosingBrace) {
                            expectedIndentLevel += 1;
                        }
                        break;
                    case 'once-at-root-twice-in-block':
                        if (node.parent === node.root()) {
                            if (precedesClosingParenthesis && !indentClosingBrace) {
                                expectedIndentLevel -= 1;
                            }
                            break;
                        }
                        if (!precedesClosingParenthesis || indentClosingBrace) {
                            expectedIndentLevel += 1;
                        }
                        break;
                    default:
                        if (precedesClosingParenthesis && !indentClosingBrace) {
                            expectedIndentLevel -= 1;
                        }
                }
            }
            const afterNewlineSpaceMatches = /^([ \t]*)\S/.exec(source.slice(match.startIndex + 1));
            if (!afterNewlineSpaceMatches) {
                return;
            }
            const afterNewlineSpace = afterNewlineSpaceMatches[1] || '';
            const expectedIndentation = indentChar.repeat(expectedIndentLevel > 0 ? expectedIndentLevel : 0);
            if (afterNewlineSpace !== expectedIndentation) {
                if (context.fix) {
                    fixPositions.unshift({
                        expectedIndentation,
                        currentIndentation: afterNewlineSpace,
                        startIndex: match.startIndex,
                    });
                }
                else {
                    report({
                        message: messages.expected(legibleExpectation(expectedIndentLevel)),
                        node,
                        index: match.startIndex + afterNewlineSpace.length + 1,
                        result,
                        ruleName,
                    });
                }
            }
        });
        if (fixPositions.length) {
            if (isRule(node)) {
                for (const fixPosition of fixPositions) {
                    node.selector = replaceIndentation(node.selector, fixPosition.currentIndentation, fixPosition.expectedIndentation, fixPosition.startIndex);
                }
            }
            if (isDeclaration(node)) {
                const declProp = node.prop;
                const declBetween = node.raws.between;
                if (!isString(declBetween)) {
                    throw new TypeError('The `between` property must be a string');
                }
                for (const fixPosition of fixPositions) {
                    if (fixPosition.startIndex < declProp.length + declBetween.length) {
                        node.raws.between = replaceIndentation(declBetween, fixPosition.currentIndentation, fixPosition.expectedIndentation, fixPosition.startIndex - declProp.length);
                    }
                    else {
                        node.value = replaceIndentation(node.value, fixPosition.currentIndentation, fixPosition.expectedIndentation, fixPosition.startIndex - declProp.length - declBetween.length);
                    }
                }
            }
            if (isAtRule(node)) {
                const atRuleName = node.name;
                const atRuleAfterName = node.raws.afterName;
                const atRuleParams = node.params;
                if (!isString(atRuleAfterName)) {
                    throw new TypeError('The `afterName` property must be a string');
                }
                for (const fixPosition of fixPositions) {
                    if (fixPosition.startIndex < 1 + atRuleName.length + atRuleAfterName.length) {
                        node.raws.afterName = replaceIndentation(atRuleAfterName, fixPosition.currentIndentation, fixPosition.expectedIndentation, fixPosition.startIndex - atRuleName.length - 1);
                    }
                    else {
                        node.params = replaceIndentation(atRuleParams, fixPosition.currentIndentation, fixPosition.expectedIndentation, fixPosition.startIndex - atRuleName.length - atRuleAfterName.length - 1);
                    }
                }
            }
        }
    }
};
function getRootBaseIndentLevel(root, baseIndentLevel, space) {
    const document = getDocument(root);
    if (!document) {
        return 0;
    }
    if (!root.source) {
        throw new Error('The root node must have a source');
    }
    const { source } = root;
    const indentLevel = source.baseIndentLevel;
    if (isNumber(indentLevel) && Number.isSafeInteger(indentLevel)) {
        return indentLevel;
    }
    const newIndentLevel = inferRootIndentLevel(root, baseIndentLevel, () => inferDocIndentSize(document, space));
    source.baseIndentLevel = newIndentLevel;
    return newIndentLevel;
}
function getDocument(node) {
    const { document } = node;
    if (document) {
        return document;
    }
    const root = node.root();
    return root && root.document;
}
function inferDocIndentSize(document, space) {
    if (!document.source)
        throw new Error('The document node must have a source');
    const docSource = document.source;
    let { indentSize } = docSource;
    if (isNumber(indentSize) && Number.isSafeInteger(indentSize)) {
        return indentSize;
    }
    const source = document.source.input.css;
    const indents = source.match(/^ *(?=\S)/gm);
    if (indents) {
        const scores = new Map();
        let lastIndentSize = 0;
        let lastLeadingSpacesLength = 0;
        const vote = (leadingSpacesLength) => {
            if (leadingSpacesLength) {
                lastIndentSize = Math.abs(leadingSpacesLength - lastLeadingSpacesLength) || lastIndentSize;
                if (lastIndentSize > 1) {
                    const score = scores.get(lastIndentSize);
                    if (score) {
                        scores.set(lastIndentSize, score + 1);
                    }
                    else {
                        scores.set(lastIndentSize, 1);
                    }
                }
            }
            else {
                lastIndentSize = 0;
            }
            lastLeadingSpacesLength = leadingSpacesLength;
        };
        for (const leadingSpaces of indents) {
            vote(leadingSpaces.length);
        }
        let bestScore = 0;
        for (const [indentSizeDate, score] of scores.entries()) {
            if (score > bestScore) {
                bestScore = score;
                indentSize = indentSizeDate;
            }
        }
    }
    indentSize = Number(indentSize) || (indents && indents[0] && indents[0].length) || Number(space) || 2;
    docSource.indentSize = indentSize;
    return indentSize;
}
function inferRootIndentLevel(root, baseIndentLevel, indentSize) {
    function getIndentLevel(indent) {
        const tabMatch = indent.match(/\t/g);
        const tabCount = tabMatch ? tabMatch.length : 0;
        const spaceMatch = indent.match(/ /g);
        const spaceCount = spaceMatch ? Math.round(spaceMatch.length / indentSize()) : 0;
        return tabCount + spaceCount;
    }
    let newBaseIndentLevel = 0;
    if (!isNumber(baseIndentLevel) || !Number.isSafeInteger(baseIndentLevel)) {
        if (!root.source)
            throw new Error('The root node must have a source');
        let source = root.source.input.css;
        source = source.replace(/^[^\r\n]+/, (firstLine) => {
            const match = root.raws.codeBefore && /(?:^|\n)([ \t]*)$/.exec(root.raws.codeBefore);
            if (match) {
                return match[1] + firstLine;
            }
            return '';
        });
        const indents = source.match(/^[ \t]*(?=\S)/gm);
        if (indents) {
            return Math.min(...indents.map((indent) => getIndentLevel(indent)));
        }
        newBaseIndentLevel = 1;
    }
    else {
        newBaseIndentLevel = baseIndentLevel;
    }
    const indents = [];
    const foundIndents = root.raws.codeBefore && /(?:^|\n)([ \t]*)\S/m.exec(root.raws.codeBefore);
    if (foundIndents) {
        let shortest = Number.MAX_SAFE_INTEGER;
        let i = 0;
        while (++i < foundIndents.length) {
            const foundIndent = foundIndents[i];
            assertString(foundIndent);
            const current = getIndentLevel(foundIndent);
            if (current < shortest) {
                shortest = current;
                if (shortest === 0) {
                    break;
                }
            }
        }
        if (shortest !== Number.MAX_SAFE_INTEGER) {
            indents.push(new Array(shortest).fill(' ').join(''));
        }
    }
    const { after } = root.raws;
    if (after) {
        let afterEnd;
        if (after.endsWith('\n')) {
            const { document } = root;
            if (document) {
                const nextRoot = document.nodes[document.nodes.indexOf(root) + 1];
                afterEnd = nextRoot ? nextRoot.raws.codeBefore : document.raws.codeAfter;
            }
            else {
                const { parent } = root;
                if (!parent)
                    throw new Error('The root node must have a parent');
                const nextRoot = parent.nodes[parent.nodes.indexOf(root) + 1];
                afterEnd = nextRoot ? nextRoot.raws.codeBefore : root.raws.codeAfter;
            }
        }
        else {
            afterEnd = after;
        }
        if (afterEnd)
            indents.push(afterEnd.match(/^[ \t]*/)[0]);
    }
    if (indents.length) {
        return Math.max(...indents.map((indent) => getIndentLevel(indent))) + newBaseIndentLevel;
    }
    return newBaseIndentLevel;
}
function fixIndentation(str, whitespace) {
    if (!isString(str)) {
        return str;
    }
    return str.replace(/\n[ \t]*(?=\S|$)/g, `\n${whitespace}`);
}
function replaceIndentation(input, searchString, replaceString, startIndex) {
    const offset = startIndex + 1;
    const stringStart = input.slice(0, offset);
    const stringEnd = input.slice(offset + searchString.length);
    return stringStart + replaceString + stringEnd;
}
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
