"use strict";
const { createPlugin } = require('stylelint');
const styleSearch = require('style-search');
const atRuleParamIndex = require('stylelint/lib/utils/atRuleParamIndex');
const declarationValueIndex = require('stylelint/lib/utils/declarationValueIndex');
const getDeclarationValue = require('stylelint/lib/utils/getDeclarationValue');
const isWhitespace = require('stylelint/lib/utils/isWhitespace');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const setDeclarationValue = require('stylelint/lib/utils/setDeclarationValue');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const ruleName = 'stylistic/function-whitespace-after';
const messages = ruleMessages(ruleName, {
    expected: 'Expected whitespace after ")"',
    rejected: 'Unexpected whitespace after ")"',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/function-whitespace-after',
    fixable: true,
};
const ACCEPTABLE_AFTER_CLOSING_PAREN = new Set([')', ',', '}', ':', '/', undefined]);
const rule = (primary, _secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
        actual: primary,
        possible: ['always', 'never'],
    });
    if (!validOptions) {
        return;
    }
    function check(node, value, nodeIndex, fix) {
        styleSearch({
            source: value,
            target: ')',
            functionArguments: 'only',
        }, (match) => {
            checkClosingParen(value, match.startIndex + 1, node, nodeIndex, fix);
        });
    }
    function checkClosingParen(source, index, node, nodeIndex, fix) {
        const nextChar = source.charAt(index);
        if (!nextChar)
            return;
        if (primary === 'always') {
            if (nextChar === ' ') {
                return;
            }
            if (nextChar === '\n') {
                return;
            }
            if (source.slice(index, index + 2) === '\r\n') {
                return;
            }
            if (ACCEPTABLE_AFTER_CLOSING_PAREN.has(nextChar)) {
                return;
            }
            if (fix) {
                fix(index);
                return;
            }
            report({
                message: messages.expected,
                node,
                index: nodeIndex + index,
                result,
                ruleName,
            });
        }
        else if (primary === 'never' && isWhitespace(nextChar)) {
            if (fix) {
                fix(index);
                return;
            }
            report({
                message: messages.rejected,
                node,
                index: nodeIndex + index,
                result,
                ruleName,
            });
        }
    }
    function createFixer(value) {
        let fixed = '';
        let lastIndex = 0;
        let applyFix;
        if (primary === 'always') {
            applyFix = (index) => {
                fixed += value.slice(lastIndex, index) + ' ';
                lastIndex = index;
            };
        }
        else if (primary === 'never') {
            applyFix = (index) => {
                let whitespaceEndIndex = index + 1;
                while (whitespaceEndIndex < value.length
                    && isWhitespace(value.charAt(whitespaceEndIndex))) {
                    whitespaceEndIndex += 1;
                }
                fixed += value.slice(lastIndex, index);
                lastIndex = whitespaceEndIndex;
            };
        }
        else {
            throw new Error(`Unexpected option: "${primary}"`);
        }
        return {
            applyFix,
            get hasFixed() {
                return Boolean(lastIndex);
            },
            get fixed() {
                return fixed + value.slice(lastIndex);
            },
        };
    }
    root.walkAtRules(/^import$/i, (atRule) => {
        const param = (atRule.raws.params && atRule.raws.params.raw) || atRule.params;
        const fixer = context.fix && createFixer(param);
        check(atRule, param, atRuleParamIndex(atRule), fixer ? fixer.applyFix : undefined);
        if (fixer && fixer.hasFixed) {
            if (atRule.raws.params) {
                atRule.raws.params.raw = fixer.fixed;
            }
            else {
                atRule.params = fixer.fixed;
            }
        }
    });
    root.walkDecls((decl) => {
        const value = getDeclarationValue(decl);
        const fixer = context.fix && createFixer(value);
        check(decl, value, declarationValueIndex(decl), fixer ? fixer.applyFix : undefined);
        if (fixer && fixer.hasFixed) {
            setDeclarationValue(decl, fixer.fixed);
        }
    });
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
