"use strict";
const { createPlugin } = require('stylelint');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const whitespaceChecker = require('stylelint/lib/utils/whitespaceChecker');
const functionCommaSpaceChecker = require('stylelint/lib/rules/functionCommaSpaceChecker');
const fixer = require('stylelint/lib/rules/functionCommaSpaceFix');
const ruleName = 'stylistic/function-comma-newline-before';
const messages = ruleMessages(ruleName, {
    expectedBefore: () => 'Expected newline before ","',
    expectedBeforeMultiLine: () => 'Expected newline before "," in a multi-line function',
    rejectedBeforeMultiLine: () => 'Unexpected whitespace before "," in a multi-line function',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/function-comma-newline-before',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => {
    const checker = whitespaceChecker('newline', primary, messages);
    return (root, result) => {
        const validOptions = validateOptions(result, ruleName, {
            actual: primary,
            possible: ['always', 'always-multi-line', 'never-multi-line'],
        });
        if (!validOptions) {
            return;
        }
        functionCommaSpaceChecker({
            root,
            result,
            locationChecker: checker.beforeAllowingIndentation,
            checkedRuleName: ruleName,
            fix: context.fix
                ? (div, index, nodes) => fixer({
                    div,
                    index,
                    nodes,
                    expectation: primary,
                    position: 'before',
                    symb: context.newline || '',
                })
                : null,
        });
    };
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
