"use strict";
const { createPlugin } = require('stylelint');
const declarationValueIndex = require('stylelint/lib/utils/declarationValueIndex');
const isStandardSyntaxDeclaration = require('stylelint/lib/utils/isStandardSyntaxDeclaration');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const whitespaceChecker = require('stylelint/lib/utils/whitespaceChecker');
const ruleName = 'stylistic/declaration-colon-newline-after';
const messages = ruleMessages(ruleName, {
    expectedAfter: () => 'Expected newline after ":"',
    expectedAfterMultiLine: () => 'Expected newline after ":" with a multi-line declaration',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/declaration-colon-newline-after',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => {
    const checker = whitespaceChecker('newline', primary, messages);
    return (root, result) => {
        const validOptions = validateOptions(result, ruleName, {
            actual: primary,
            possible: ['always', 'always-multi-line'],
        });
        if (!validOptions) {
            return;
        }
        root.walkDecls((decl) => {
            if (!isStandardSyntaxDeclaration(decl)) {
                return;
            }
            const endOfPropIndex = declarationValueIndex(decl) + (decl.raws.between || '').length - 1;
            const propPlusColon = `${decl.toString().slice(0, endOfPropIndex)}xxx`;
            for (let i = 0, l = propPlusColon.length; i < l; i += 1) {
                if (propPlusColon[i] !== ':') {
                    continue;
                }
                const indexToCheck = /^[^\S\r\n]*\/\*/.test(propPlusColon.slice(i + 1))
                    ? propPlusColon.indexOf('*/', i) + 1
                    : i;
                checker.afterOneOnly({
                    source: propPlusColon,
                    index: indexToCheck,
                    lineCheckStr: decl.value,
                    err: (m) => {
                        if (context.fix) {
                            const { between } = decl.raws;
                            if (between == null)
                                throw new Error('`between` must be present');
                            const betweenStart = declarationValueIndex(decl) - between.length;
                            const sliceIndex = indexToCheck - betweenStart + 1;
                            const betweenBefore = between.slice(0, sliceIndex);
                            const betweenAfter = between.slice(sliceIndex);
                            decl.raws.between = /^\s*\n/.test(betweenAfter)
                                ? betweenBefore + betweenAfter.replace(/^[^\S\r\n]*/, '')
                                : betweenBefore + context.newline + betweenAfter;
                            return;
                        }
                        report({
                            message: m,
                            node: decl,
                            index: indexToCheck,
                            result,
                            ruleName,
                        });
                    },
                });
            }
        });
    };
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
