"use strict";
const { createPlugin } = require('stylelint');
const blockString = require('stylelint/lib/utils/blockString');
const rawNodeString = require('stylelint/lib/utils/rawNodeString');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const whitespaceChecker = require('stylelint/lib/utils/whitespaceChecker');
const { isAtRule, isRule } = require('stylelint/lib/utils/typeGuards');
const ruleName = 'stylistic/declaration-block-semicolon-space-after';
const messages = ruleMessages(ruleName, {
    expectedAfter: () => 'Expected single space after ";"',
    rejectedAfter: () => 'Unexpected whitespace after ";"',
    expectedAfterSingleLine: () => 'Expected single space after ";" in a single-line declaration block',
    rejectedAfterSingleLine: () => 'Unexpected whitespace after ";" in a single-line declaration block',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/declaration-block-semicolon-space-after',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => {
    const checker = whitespaceChecker('space', primary, messages);
    return (root, result) => {
        const validOptions = validateOptions(result, ruleName, {
            actual: primary,
            possible: ['always', 'never', 'always-single-line', 'never-single-line'],
        });
        if (!validOptions) {
            return;
        }
        root.walkDecls((decl) => {
            const parentRule = decl.parent;
            if (!parentRule)
                throw new Error('A parent node must be present');
            if (!isAtRule(parentRule) && !isRule(parentRule)) {
                return;
            }
            if (!parentRule.raws.semicolon && parentRule.last === decl) {
                return;
            }
            const nextDecl = decl.next();
            if (!nextDecl) {
                return;
            }
            checker.after({
                source: rawNodeString(nextDecl),
                index: -1,
                lineCheckStr: blockString(parentRule),
                err: (m) => {
                    if (context.fix) {
                        if (primary.startsWith('always')) {
                            nextDecl.raws.before = ' ';
                            return;
                        }
                        if (primary.startsWith('never')) {
                            nextDecl.raws.before = '';
                            return;
                        }
                    }
                    report({
                        message: m,
                        node: decl,
                        index: decl.toString().length + 1,
                        result,
                        ruleName,
                    });
                },
            });
        });
    };
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
