"use strict";
const { createPlugin } = require('stylelint');
const blockString = require('stylelint/lib/utils/blockString');
const nextNonCommentNode = require('stylelint/lib/utils/nextNonCommentNode');
const rawNodeString = require('stylelint/lib/utils/rawNodeString');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const whitespaceChecker = require('stylelint/lib/utils/whitespaceChecker');
const { isAtRule, isRule } = require('stylelint/lib/utils/typeGuards');
const ruleName = 'stylistic/declaration-block-semicolon-newline-after';
const messages = ruleMessages(ruleName, {
    expectedAfter: () => 'Expected newline after ";"',
    expectedAfterMultiLine: () => 'Expected newline after ";" in a multi-line declaration block',
    rejectedAfterMultiLine: () => 'Unexpected newline after ";" in a multi-line declaration block',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/declaration-block-semicolon-newline-after',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => {
    const checker = whitespaceChecker('newline', primary, messages);
    return (root, result) => {
        const validOptions = validateOptions(result, ruleName, {
            actual: primary,
            possible: ['always', 'always-multi-line', 'never-multi-line'],
        });
        if (!validOptions) {
            return;
        }
        root.walkDecls((decl) => {
            const parentRule = decl.parent;
            if (!parentRule)
                throw new Error('A parent node must be present');
            if (!isAtRule(parentRule) && !isRule(parentRule)) {
                return;
            }
            if (!parentRule.raws.semicolon && parentRule.last === decl) {
                return;
            }
            const nextNode = decl.next();
            if (!nextNode) {
                return;
            }
            const nodeToCheck = nextNonCommentNode(nextNode);
            if (!nodeToCheck) {
                return;
            }
            checker.afterOneOnly({
                source: rawNodeString(nodeToCheck),
                index: -1,
                lineCheckStr: blockString(parentRule),
                err: (m) => {
                    if (context.fix) {
                        if (primary.startsWith('always')) {
                            const index = nodeToCheck.raws.before.search(/\r?\n/);
                            nodeToCheck.raws.before = index >= 0
                                ? nodeToCheck.raws.before.slice(index)
                                : context.newline + nodeToCheck.raws.before;
                            return;
                        }
                        if (primary === 'never-multi-line') {
                            nodeToCheck.raws.before = '';
                            return;
                        }
                    }
                    report({
                        message: m,
                        node: decl,
                        index: decl.toString().length + 1,
                        result,
                        ruleName,
                    });
                },
            });
        });
    };
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
