"use strict";
const { createPlugin } = require('stylelint');
const beforeBlockString = require('stylelint/lib/utils/beforeBlockString');
const blockString = require('stylelint/lib/utils/blockString');
const hasBlock = require('stylelint/lib/utils/hasBlock');
const hasEmptyBlock = require('stylelint/lib/utils/hasEmptyBlock');
const optionsMatches = require('stylelint/lib/utils/optionsMatches');
const rawNodeString = require('stylelint/lib/utils/rawNodeString');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const whitespaceChecker = require('stylelint/lib/utils/whitespaceChecker');
const ruleName = 'stylistic/block-opening-brace-newline-after';
const messages = ruleMessages(ruleName, {
    expectedAfter: () => 'Expected newline after "{"',
    expectedAfterMultiLine: () => 'Expected newline after "{" of a multi-line block',
    rejectedAfterMultiLine: () => 'Unexpected whitespace after "{" of a multi-line block',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/block-opening-brace-newline-after',
    fixable: true,
};
const rule = (primary, secondaryOptions, context) => {
    const checker = whitespaceChecker('newline', primary, messages);
    return (root, result) => {
        const validOptions = validateOptions(result, ruleName, {
            actual: primary,
            possible: ['always', 'rules', 'always-multi-line', 'never-multi-line'],
        }, {
            actual: secondaryOptions,
            possible: {
                ignore: ['rules'],
            },
            optional: true,
        });
        if (!validOptions) {
            return;
        }
        if (!optionsMatches(secondaryOptions, 'ignore', 'rules')) {
            root.walkRules(check);
        }
        root.walkAtRules(check);
        function check(statement) {
            if (!hasBlock(statement) || hasEmptyBlock(statement)) {
                return;
            }
            const backupCommentNextBefores = new Map();
            function nextNode(startNode) {
                if (!startNode || !startNode.next)
                    return;
                if (startNode.type === 'comment') {
                    const reNewLine = /\r?\n/;
                    const newLineMatch = reNewLine.test(startNode.raws.before || '');
                    const next = startNode.next();
                    if (next && newLineMatch && !reNewLine.test(next.raws.before || '')) {
                        backupCommentNextBefores.set(next, next.raws.before);
                        next.raws.before = startNode.raws.before;
                    }
                    return nextNode(next);
                }
                return startNode;
            }
            const nodeToCheck = nextNode(statement.first);
            if (!nodeToCheck) {
                return;
            }
            checker.afterOneOnly({
                source: rawNodeString(nodeToCheck),
                index: -1,
                lineCheckStr: blockString(statement),
                err: (m) => {
                    if (context.fix) {
                        const nodeToCheckRaws = nodeToCheck.raws;
                        if (typeof nodeToCheckRaws.before !== 'string')
                            return;
                        if (primary.startsWith('always')) {
                            const index = nodeToCheckRaws.before.search(/\r?\n/);
                            nodeToCheckRaws.before = index >= 0
                                ? nodeToCheckRaws.before.slice(index)
                                : context.newline + nodeToCheckRaws.before;
                            backupCommentNextBefores.delete(nodeToCheck);
                            return;
                        }
                        if (primary === 'never-multi-line') {
                            for (const [node, before] of backupCommentNextBefores.entries()) {
                                node.raws.before = before;
                            }
                            backupCommentNextBefores.clear();
                            const reNewLine = /\r?\n/;
                            let fixTarget = statement.first;
                            while (fixTarget) {
                                const fixTargetRaws = fixTarget.raws;
                                if (typeof fixTargetRaws.before !== 'string')
                                    continue;
                                if (reNewLine.test(fixTargetRaws.before || '')) {
                                    fixTargetRaws.before = fixTargetRaws.before.replace(/\r?\n/g, '');
                                }
                                if (fixTarget.type !== 'comment') {
                                    break;
                                }
                                fixTarget = fixTarget.next();
                            }
                            nodeToCheckRaws.before = '';
                            return;
                        }
                    }
                    report({
                        message: m,
                        node: statement,
                        index: beforeBlockString(statement, { noRawBefore: true }).length + 1,
                        result,
                        ruleName,
                    });
                },
            });
            for (const [node, before] of backupCommentNextBefores.entries()) {
                node.raws.before = before;
            }
        }
    };
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
