"use strict";
const { createPlugin } = require('stylelint');
const blockString = require('stylelint/lib/utils/blockString');
const hasBlock = require('stylelint/lib/utils/hasBlock');
const rawNodeString = require('stylelint/lib/utils/rawNodeString');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const whitespaceChecker = require('stylelint/lib/utils/whitespaceChecker');
const ruleName = 'stylistic/block-closing-brace-space-after';
const messages = ruleMessages(ruleName, {
    expectedAfter: () => 'Expected single space after "}"',
    rejectedAfter: () => 'Unexpected whitespace after "}"',
    expectedAfterSingleLine: () => 'Expected single space after "}" of a single-line block',
    rejectedAfterSingleLine: () => 'Unexpected whitespace after "}" of a single-line block',
    expectedAfterMultiLine: () => 'Expected single space after "}" of a multi-line block',
    rejectedAfterMultiLine: () => 'Unexpected whitespace after "}" of a multi-line block',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/block-closing-brace-space-after',
};
const rule = (primary) => {
    const checker = whitespaceChecker('space', primary, messages);
    return (root, result) => {
        const validOptions = validateOptions(result, ruleName, {
            actual: primary,
            possible: [
                'always',
                'never',
                'always-single-line',
                'never-single-line',
                'always-multi-line',
                'never-multi-line',
            ],
        });
        if (!validOptions) {
            return;
        }
        root.walkRules(check);
        root.walkAtRules(check);
        function check(statement) {
            const nextNode = statement.next();
            if (!nextNode) {
                return;
            }
            if (!hasBlock(statement)) {
                return;
            }
            let reportIndex = statement.toString().length;
            let source = rawNodeString(nextNode);
            if (source && source.startsWith(';')) {
                source = source.slice(1);
                reportIndex += 1;
            }
            checker.after({
                source,
                index: -1,
                lineCheckStr: blockString(statement),
                err: (msg) => {
                    report({
                        message: msg,
                        node: statement,
                        index: reportIndex,
                        result,
                        ruleName,
                    });
                },
            });
        }
    };
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
