"use strict";
const { createPlugin } = require('stylelint');
const blockString = require('stylelint/lib/utils/blockString');
const hasBlock = require('stylelint/lib/utils/hasBlock');
const hasEmptyBlock = require('stylelint/lib/utils/hasEmptyBlock');
const isSingleLineString = require('stylelint/lib/utils/isSingleLineString');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const ruleName = 'stylistic/block-closing-brace-newline-before';
const messages = ruleMessages(ruleName, {
    expectedBefore: 'Expected newline before "}"',
    expectedBeforeMultiLine: 'Expected newline before "}" of a multi-line block',
    rejectedBeforeMultiLine: 'Unexpected whitespace before "}" of a multi-line block',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/block-closing-brace-newline-before',
    fixable: true,
};
const rule = (primary, _secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
        actual: primary,
        possible: ['always', 'always-multi-line', 'never-multi-line'],
    });
    if (!validOptions) {
        return;
    }
    root.walkRules(check);
    root.walkAtRules(check);
    function check(statement) {
        if (!hasBlock(statement) || hasEmptyBlock(statement)) {
            return;
        }
        const after = (statement.raws.after || '').replace(/;+/, '');
        if (after === undefined) {
            return;
        }
        const blockIsMultiLine = !isSingleLineString(blockString(statement));
        const statementString = statement.toString();
        let index = statementString.length - 2;
        if (statementString[index - 1] === '\r') {
            index -= 1;
        }
        if (!after.startsWith('\n') && !after.startsWith('\r\n')) {
            if (primary === 'always') {
                complain(messages.expectedBefore);
            }
            else if (blockIsMultiLine && primary === 'always-multi-line') {
                complain(messages.expectedBeforeMultiLine);
            }
        }
        if (after !== '' && blockIsMultiLine && primary === 'never-multi-line') {
            complain(messages.rejectedBeforeMultiLine);
        }
        function complain(message) {
            if (context.fix) {
                const statementRaws = statement.raws;
                if (typeof statementRaws.after !== 'string')
                    return;
                if (primary.startsWith('always')) {
                    const firstWhitespaceIndex = statementRaws.after.search(/\s/);
                    const newlineBefore = firstWhitespaceIndex >= 0
                        ? statementRaws.after.slice(0, firstWhitespaceIndex)
                        : statementRaws.after;
                    const newlineAfter = firstWhitespaceIndex >= 0 ? statementRaws.after.slice(firstWhitespaceIndex) : '';
                    const newlineIndex = newlineAfter.search(/\r?\n/);
                    statementRaws.after = newlineIndex >= 0
                        ? newlineBefore + newlineAfter.slice(newlineIndex)
                        : newlineBefore + context.newline + newlineAfter;
                    return;
                }
                if (primary === 'never-multi-line') {
                    statementRaws.after = statementRaws.after.replace(/\s/g, '');
                    return;
                }
            }
            report({
                message,
                result,
                ruleName,
                node: statement,
                index,
            });
        }
    }
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
