"use strict";
const { createPlugin } = require('stylelint');
const addEmptyLineAfter = require('stylelint/lib/utils/addEmptyLineAfter');
const blockString = require('stylelint/lib/utils/blockString');
const hasBlock = require('stylelint/lib/utils/hasBlock');
const hasEmptyBlock = require('stylelint/lib/utils/hasEmptyBlock');
const hasEmptyLine = require('stylelint/lib/utils/hasEmptyLine');
const isSingleLineString = require('stylelint/lib/utils/isSingleLineString');
const optionsMatches = require('stylelint/lib/utils/optionsMatches');
const removeEmptyLinesAfter = require('stylelint/lib/utils/removeEmptyLinesAfter');
const report = require('stylelint/lib/utils/report');
const ruleMessages = require('stylelint/lib/utils/ruleMessages');
const validateOptions = require('stylelint/lib/utils/validateOptions');
const ruleName = 'stylistic/block-closing-brace-empty-line-before';
const messages = ruleMessages(ruleName, {
    expected: 'Expected empty line before closing brace',
    rejected: 'Unexpected empty line before closing brace',
});
const meta = {
    url: 'https://github.com/elirasza/stylelint-stylistic/tree/main/lib/rules/block-closing-brace-empty-line-before',
    fixable: true,
};
const rule = (primary, secondaryOptions, context) => (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
        actual: primary,
        possible: ['always-multi-line', 'never'],
    }, {
        actual: secondaryOptions,
        possible: {
            except: ['after-closing-brace'],
        },
        optional: true,
    });
    if (!validOptions) {
        return;
    }
    root.walkRules(check);
    root.walkAtRules(check);
    function check(statement) {
        if (!hasBlock(statement) || hasEmptyBlock(statement)) {
            return;
        }
        const before = (statement.raws.after || '').replace(/;+/, '');
        const statementString = statement.toString();
        let index = statementString.length - 1;
        if (statementString[index - 1] === '\r') {
            index -= 1;
        }
        const expectEmptyLineBefore = (() => {
            const childNodeTypes = statement.nodes.map((item) => item.type);
            if (optionsMatches(secondaryOptions, 'except', 'after-closing-brace')
                && statement.type === 'atrule'
                && !childNodeTypes.includes('decl')) {
                return primary === 'never';
            }
            return primary === 'always-multi-line' && !isSingleLineString(blockString(statement));
        })();
        const hasEmptyLineBefore = hasEmptyLine(before);
        if (expectEmptyLineBefore === hasEmptyLineBefore) {
            return;
        }
        if (context.fix) {
            const { newline } = context;
            if (typeof newline !== 'string')
                return;
            if (expectEmptyLineBefore) {
                addEmptyLineAfter(statement, newline);
            }
            else {
                removeEmptyLinesAfter(statement, newline);
            }
            return;
        }
        const message = expectEmptyLineBefore ? messages.expected : messages.rejected;
        report({
            message,
            result,
            ruleName,
            node: statement,
            index,
        });
    }
};
rule.ruleName = ruleName;
rule.messages = messages;
rule.meta = meta;
module.exports = { messages, meta, plugin: createPlugin(ruleName, rule), rule, ruleName };
