<?php

namespace Drupal\emory_isss_core;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\node\NodeInterface;

/**
 * Provides a service to get node hierarchy list.
 */
class NodeHierarchyService {


  /**
   * The entity manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs a new NodeHierarchyService.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The current route match.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, RouteMatchInterface $routeMatch) {
    $this->entityTypeManager = $entityTypeManager;
    $this->routeMatch = $routeMatch;
  }

  /**
   * Function to get the root parent node id.
   *
   * @param int $nid
   *   Node id value.
   *
   * @return int|string|null
   *   Returns parent node id.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getRootParent($nid = NULL) {
    if (!$nid) {
      $nid = $this->getCurrentNodeId();
    }
    $current_node = $this->entityTypeManager->getStorage('node')->load($nid);

    while ($current_node && $current_node->get('field_parent_content')->target_id) {
      $parent_node = $this->entityTypeManager->getStorage('node')->load($current_node->get('field_parent_content')->target_id);

      if ($parent_node) {
        $current_node = $parent_node;
      }
      else {
        break;
      }
    }

    return $current_node->id();
  }

  /**
   * Function to get the current node id.
   *
   * @return int|string|null
   *   Returns current node id if it is available.
   */
  protected function getCurrentNodeId() {
    $node = $this->routeMatch->getParameter('node');
    if ($node && $node instanceof NodeInterface) {
      return $node->id();
    }
    return NULL;
  }

  /**
   * Function to get the children tree.
   *
   * @param int $parent_node_id
   *   Parent node id.
   * @param int $max_depth
   *   Max depth value.
   * @param int $current_depth
   *   Current depth value.
   *
   * @return array
   *   Returns an array.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  public function getChildrenTree($parent_node_id, $max_depth = 8, $current_depth = 1) {
    $tree = [];
    if ($current_depth == 1) {
      $node = $this->entityTypeManager->getStorage('node')->load($parent_node_id);
      $tree[0] = [
        'node' => [
          'id' => $parent_node_id,
          'label' => $node->label(),
          'url' => $node->toUrl()->toString(),
        ],
      ];
    }
    $query = $this->entityTypeManager->getStorage('node')
      ->getQuery()
      ->condition('field_parent_content', $parent_node_id)
      ->accessCheck(FALSE);
    $child_node_ids = $query->execute();
    foreach ($child_node_ids as $child_node_id) {
      $child_node = $this->entityTypeManager->getStorage('node')->load($child_node_id);
      if ($current_depth < $max_depth) {
        if ($current_depth == 1) {
          $tree[0]['children'][] = [
            'node' => [
              'id' => $child_node->id(),
              'label' => $child_node->label(),
              'url' => $child_node->toUrl()->toString(),
            ],
            'children' => $this->getChildrenTree($child_node_id, $max_depth, $current_depth + 1),
          ];
        }
        else {
          $tree[] = [
            'node' => [
              'id' => $child_node->id(),
              'label' => $child_node->label(),
              'url' => $child_node->toUrl()->toString(),
            ],
            'children' => $this->getChildrenTree($child_node_id, $max_depth, $current_depth + 1),
          ];
        }
      }
    }
    return $tree;

  }

}
