<?php

namespace CommerceGuys\Addressing\Subdivision;

/**
 * Provides logic for updating address values when subdivisions change in the library.
 *
 * Primarily covers the Addressing 1.x => 2.x updates, where subdivisions started
 * being keyed by ISO code where available.
 *
 * Old administrative_area and locality values can be updated one by one via getUpdatedValue(),
 * or the full update map can be retrieved via getUpdateMap() and converted to update queries.
 */
class SubdivisionUpdater
{
    /**
     * Updates the given value (administrative area, locality, or dependent locality).
     *
     * @return string The updated value, or the given value if no updates were found.
     */
    public static function updateValue(string $countryCode, string $value): string
    {
        // A list of all country codes defined in getUpdateMap(), duplicated here
        // to avoid retrieving the entire map for countries that don't need it.
        $countryCodes = [
            'AD', 'AE', 'AM', 'AR', 'BS', 'CL', 'CN', 'CU', 'EG', 'ES', 'ID',
            'IE', 'IN', 'IR', 'JM', 'JP', 'KN', 'KP', 'KR', 'MX', 'MY', 'MZ',
            'NG', 'NI', 'NR', 'PE', 'PH', 'RU', 'SR', 'SV', 'TH', 'TR', 'TV',
            'TW', 'UA', 'UY', 'VE', 'VN',
        ];
        if (in_array($countryCode, $countryCodes) && strlen($value) > 2) {
            $map = self::getUpdateMap();
            if (isset($map[$countryCode][$value])) {
                return $map[$countryCode][$value];
            }
        }

        return $value;
    }

    /**
     * Gets the update map.
     *
     * @return array The update map, keyed by country code.
     */
    public static function getUpdateMap(): array
    {
        return [
            'AD' => [
              'Parròquia d\'Andorra la Vella' => '07',
              'Canillo' => '02',
              'Encamp' => '03',
              'Escaldes-Engordany' => '08',
              'La Massana' => '04',
              'Ordino' => '05',
              'Sant Julià de Lòria' => '06',
            ],
            'AE' => [
              'Abu Dhabi' => 'AZ',
              'Sharjah' => 'SH',
              'Fujairah' => 'FU',
              'Umm Al Quwain' => 'UQ',
              'Dubai' => 'DU',
              'Ras al Khaimah' => 'RK',
              'Ajman' => 'AJ',
            ],
            'AM' => [
              'Aragatsotn' => 'AG',
              'Ararat' => 'AR',
              'Armavir' => 'AV',
              'Gegharkunik' => 'GR',
              'Yerevan' => 'ER',
              'Lori' => 'LO',
              'Kotayk' => 'KT',
              'Shirak' => 'SH',
              'Syunik' => 'SU',
              'Vayots Dzor' => 'VD',
              'Tavush' => 'TV',
            ],
            'AR' => [
              'Buenos Aires' => 'B',
              'Catamarca' => 'K',
              'Chaco' => 'H',
              'Chubut' => 'U',
              'Ciudad Autónoma de Buenos Aires' => 'C',
              'Córdoba' => 'X',
              'Corrientes' => 'W',
              'Entre Ríos' => 'E',
              'Formosa' => 'P',
              'Jujuy' => 'Y',
              'La Pampa' => 'L',
              'La Rioja' => 'F',
              'Mendoza' => 'M',
              'Misiones' => 'N',
              'Neuquén' => 'Q',
              'Río Negro' => 'R',
              'Salta' => 'A',
              'San Juan' => 'J',
              'San Luis' => 'D',
              'Santa Cruz' => 'Z',
              'Santa Fe' => 'S',
              'Santiago del Estero' => 'G',
              'Tierra del Fuego' => 'V',
              'Tucumán' => 'T',
            ],
            'BS' => [
              'Acklins' => 'AK',
              'Berry Islands' => 'BY',
              'Bimini' => 'BI',
              'Black Point' => 'BP',
              'Cat Island' => 'CI',
              'Central Abaco' => 'CO',
              'Central Andros' => 'CS',
              'Central Eleuthera' => 'CE',
              'City of Freeport' => 'FP',
              'Crooked Island and Long Cay' => 'CK',
              'East Grand Bahama' => 'EG',
              'Exuma' => 'EX',
              'Grand Cay' => 'GC',
              'Harbour Island' => 'HI',
              'Hope Town' => 'HT',
              'Inagua' => 'IN',
              'Long Island' => 'LI',
              'Mangrove Cay' => 'MC',
              'Mayaguana' => 'MG',
              'Moore\'s Island' => 'MI',
              'New Providence' => 'NP',
              'North Abaco' => 'NO',
              'North Andros' => 'NS',
              'North Eleuthera' => 'NE',
              'Ragged Island' => 'RI',
              'Rum Cay' => 'RC',
              'San Salvador' => 'SS',
              'South Abaco' => 'SO',
              'South Andros' => 'SA',
              'South Eleuthera' => 'SE',
              'Spanish Wells' => 'SW',
              'West Grand Bahama' => 'WG',
            ],
            'CL' => [
              'Antofagasta' => 'AN',
              'Araucanía' => 'AR',
              'Arica y Parinacota' => 'AP',
              'Atacama' => 'AT',
              'Aysén' => 'AI',
              'Biobío' => 'BI',
              'Coquimbo' => 'CO',
              'O\'Higgins' => 'LI',
              'Los Lagos' => 'LL',
              'Los Ríos' => 'LR',
              'Magallanes' => 'MA',
              'Maule' => 'ML',
              'Región Metropolitana' => 'RM',
              'Ñuble' => 'NB',
              'Tarapacá' => 'TA',
              'Valparaíso' => 'VS',
            ],
            'CN' => [
              'Anhui Sheng' => 'AH',
              'Macau' => 'MO',
              'Beijing Shi' => 'BJ',
              'Chongqing Shi' => 'CQ',
              'Fujian Sheng' => 'FJ',
              'Gansu Sheng' => 'GS',
              'Guangdong Sheng' => 'GD',
              'Guangxi Zhuangzuzizhiqu' => 'GX',
              'Guizhou Sheng' => 'GZ',
              'Hainan Sheng' => 'HI',
              'Hebei Sheng' => 'HE',
              'Henan Sheng' => 'HA',
              'Heilongjiang Sheng' => 'HL',
              'Hubei Sheng' => 'HB',
              'Hunan Sheng' => 'HN',
              'Jilin Sheng' => 'JL',
              'Jiangsu Sheng' => 'JS',
              'Jiangxi Sheng' => 'JX',
              'Liaoning Sheng' => 'LN',
              'Neimenggu Zizhiqu' => 'NM',
              'Ningxia Huizuzizhiqu' => 'NX',
              'Qinghai Sheng' => 'QH',
              'Shandong Sheng' => 'SD',
              'Shanxi Sheng' => 'SX',
              'Shaanxi Sheng' => 'SN',
              'Shanghai Shi' => 'SH',
              'Sichuan Sheng' => 'SC',
              'Taiwan' => 'TW',
              'Tianjin Shi' => 'TJ',
              'Xizang Zizhiqu' => 'XZ',
              'Hong Kong' => 'HK',
              'Xinjiang Weiwuerzizhiqu' => 'XJ',
              'Yunnan Sheng' => 'YN',
              'Zhejiang Sheng' => 'ZJ',
            ],
            'CU' => [
              'Artemisa' => '15',
              'Camagüey' => '09',
              'Ciego de Ávila' => '08',
              'Cienfuegos' => '06',
              'Granma' => '12',
              'Guantánamo' => '14',
              'Holguín' => '11',
              'Isla de la Juventud' => '99',
              'La Habana' => '03',
              'Las Tunas' => '10',
              'Matanzas' => '04',
              'Mayabeque' => '16',
              'Pinar del Río' => '01',
              'Sancti Spíritus' => '07',
              'Santiago de Cuba' => '13',
              'Villa Clara' => '05',
            ],
            'EG' => [
              'Aswan Governorate' => 'ASN',
              'Asyut Governorate' => 'AST',
              'Alexandria Governorate' => 'ALX',
              'Ismailia Governorate' => 'IS',
              'Luxor Governorate' => 'LX',
              'Red Sea Governorate' => 'BA',
              'El Beheira Governorate' => 'BH',
              'Giza Governorate' => 'GZ',
              'Dakahlia Governorate' => 'DK',
              'Suez Governorate' => 'SUZ',
              'Ash Sharqia Governorate' => 'SHR',
              'Gharbia Governorate' => 'GH',
              'Faiyum Governorate' => 'FYM',
              'Cairo Governorate' => 'C',
              'Qalyubia Governorate' => 'KB',
              'Menofia Governorate' => 'MNF',
              'Menia Governorate' => 'MN',
              'New Valley Governorate' => 'WAD',
              'Beni Suef Governorate' => 'BNS',
              'Port Said Governorate' => 'PTS',
              'South Sinai Governorate' => 'JS',
              'Damietta Governorate' => 'DT',
              'Sohag Governorate' => 'SHG',
              'North Sinai Governorate' => 'SIN',
              'Qena Governorate' => 'KN',
              'Kafr El Sheikh Governorate' => 'KFS',
              'Matrouh Governorate' => 'MT',
            ],
            'ES' => [
              'A Coruña' => 'C',
              'Álava' => 'VI',
              'Albacete' => 'AB',
              'Alicante' => 'A',
              'Almería' => 'AL',
              'Asturias' => 'O',
              'Ávila' => 'AV',
              'Badajoz' => 'BA',
              'Balears' => 'PM',
              'Barcelona' => 'B',
              'Burgos' => 'BU',
              'Cáceres' => 'CC',
              'Cádiz' => 'CA',
              'Cantabria' => 'S',
              'Castellón' => 'CS',
              'Ceuta' => 'CE',
              'Ciudad Real' => 'CR',
              'Córdoba' => 'CO',
              'Cuenca' => 'CU',
              'Girona' => 'GI',
              'Granada' => 'GR',
              'Guadalajara' => 'GU',
              'Guipúzcoa' => 'SS',
              'Huelva' => 'H',
              'Huesca' => 'HU',
              'Jaén' => 'J',
              'La Rioja' => 'LO',
              'Las Palmas' => 'GC',
              'León' => 'LE',
              'Lleida' => 'L',
              'Lugo' => 'LU',
              'Madrid' => 'M',
              'Málaga' => 'MA',
              'Melilla' => 'ML',
              'Murcia' => 'MU',
              'Navarra' => 'NA',
              'Ourense' => 'OR',
              'Palencia' => 'P',
              'Pontevedra' => 'PO',
              'Salamanca' => 'SA',
              'Santa Cruz de Tenerife' => 'TF',
              'Segovia' => 'SG',
              'Sevilla' => 'SE',
              'Soria' => 'SO',
              'Tarragona' => 'T',
              'Teruel' => 'TE',
              'Toledo' => 'TO',
              'Valencia' => 'V',
              'Valladolid' => 'VA',
              'Vizcaya' => 'BI',
              'Zamora' => 'ZA',
              'Zaragoza' => 'Z',
            ],
            'ID' => [
              'Aceh' => 'AC',
              'Bali' => 'BA',
              'Banten' => 'BT',
              'Bengkulu' => 'BE',
              'Special Region of Yogyakarta' => 'YO',
              'Jakarta' => 'JK',
              'Gorontalo' => 'GO',
              'Jambi' => 'JA',
              'West Java' => 'JB',
              'Central Java' => 'JT',
              'East Java' => 'JI',
              'West Kalimantan' => 'KB',
              'South Kalimantan' => 'KS',
              'Central Kalimantan' => 'KT',
              'East Kalimantan' => 'KI',
              'North Kalimantan' => 'KU',
              'Bangka Belitung Islands' => 'BB',
              'Riau Islands' => 'KR',
              'Lampung' => 'LA',
              'Maluku' => 'MA',
              'North Maluku' => 'MU',
              'West Nusa Tenggara' => 'NB',
              'East Nusa Tenggara' => 'NT',
              'Papua' => 'PA',
              'West Papua' => 'PB',
              'Riau' => 'RI',
              'West Sulawesi' => 'SR',
              'South Sulawesi' => 'SN',
              'Central Sulawesi' => 'ST',
              'South East Sulawesi' => 'SG',
              'North Sulawesi' => 'SA',
              'West Sumatra' => 'SB',
              'South Sumatra' => 'SS',
              'North Sumatra' => 'SU',
            ],
            'IE' => [
              'Co. Carlow' => 'CW',
              'Co. Cavan' => 'CN',
              'Co. Clare' => 'CE',
              'Co. Cork' => 'CO',
              'Co. Donegal' => 'DL',
              'Co. Dublin' => 'D',
              'Co. Galway' => 'G',
              'Co. Kerry' => 'KY',
              'Co. Kildare' => 'KE',
              'Co. Kilkenny' => 'KK',
              'Co. Laois' => 'LS',
              'Co. Leitrim' => 'LM',
              'Co. Limerick' => 'LK',
              'Co. Longford' => 'LD',
              'Co. Louth' => 'LH',
              'Co. Mayo' => 'MO',
              'Co. Meath' => 'MH',
              'Co. Monaghan' => 'MN',
              'Co. Offaly' => 'OY',
              'Co. Roscommon' => 'RN',
              'Co. Sligo' => 'SO',
              'Co. Tipperary' => 'TA',
              'Co. Waterford' => 'WD',
              'Co. Westmeath' => 'WH',
              'Co. Wexford' => 'WX',
              'Co. Wicklow' => 'WW',
            ],
            'IN' => [
              'Andaman and Nicobar Islands' => 'AN',
              'Andhra Pradesh' => 'AP',
              'Arunachal Pradesh' => 'AR',
              'Assam' => 'AS',
              'Bihar' => 'BR',
              'Chandigarh' => 'CH',
              'Chhattisgarh' => 'CT',
              'Dadra and Nagar Haveli and Daman and Diu' => 'DH',
              'Delhi' => 'DL',
              'Goa' => 'GA',
              'Gujarat' => 'GJ',
              'Haryana' => 'HR',
              'Himachal Pradesh' => 'HP',
              'Jammu and Kashmir' => 'JK',
              'Jharkhand' => 'JH',
              'Karnataka' => 'KA',
              'Kerala' => 'KL',
              'Ladakh' => 'LA',
              'Lakshadweep' => 'LD',
              'Madhya Pradesh' => 'MP',
              'Maharashtra' => 'MH',
              'Manipur' => 'MN',
              'Meghalaya' => 'ML',
              'Mizoram' => 'MZ',
              'Nagaland' => 'NL',
              'Odisha' => 'OR',
              'Puducherry' => 'PY',
              'Punjab' => 'PB',
              'Rajasthan' => 'RJ',
              'Sikkim' => 'SK',
              'Tamil Nadu' => 'TN',
              'Telangana' => 'TG',
              'Tripura' => 'TR',
              'Uttar Pradesh' => 'UP',
              'Uttarakhand' => 'UT',
              'West Bengal' => 'WB',
            ],
            'IR' => [
              'East Azerbaijan Province' => '03',
              'West Azerbaijan Province' => '04',
              'Ardabil Province' => '24',
              'Isfahan Province' => '10',
              'Alborz Province' => '30',
              'Ilam Province' => '16',
              'Bushehr Province' => '18',
              'Tehran Province' => '23',
              'Chaharmahal and Bakhtiari Province' => '14',
              'South Khorasan Province' => '29',
              'Razavi Khorasan Province' => '09',
              'North Khorasan Province' => '28',
              'Khuzestan Province' => '06',
              'Zanjan Province' => '19',
              'Semnan Province' => '20',
              'Sistan and Baluchestan Province' => '11',
              'Fars Province' => '07',
              'Qazvin Province' => '26',
              'Qom Province' => '25',
              'Kurdistan Province' => '12',
              'Kerman Province' => '08',
              'Kermanshah Province' => '05',
              'Kohgiluyeh and Boyer-Ahmad Province' => '17',
              'Golestan Province' => '27',
              'Gilan Province' => '01',
              'Lorestan Province' => '15',
              'Mazandaran Province' => '02',
              'Markazi Province' => '00',
              'Hormozgan Province' => '22',
              'Hamadan Province' => '13',
              'Yazd Province' => '21',
            ],
            'JM' => [
              'Clarendon' => '13',
              'Hanover' => '09',
              'Kingston' => '01',
              'Manchester' => '12',
              'Portland' => '04',
              'St. Andrew' => '02',
              'St. Ann' => '06',
              'St. Catherine' => '14',
              'St. Elizabeth' => '11',
              'St. James' => '08',
              'St. Mary' => '05',
              'St. Thomas' => '03',
              'Trelawny' => '07',
              'Westmoreland' => '10',
            ],
            'JP' => [
              'Hokkaido' => '01',
              'Aomori' => '02',
              'Iwate' => '03',
              'Miyagi' => '04',
              'Akita' => '05',
              'Yamagata' => '06',
              'Fukushima' => '07',
              'Ibaraki' => '08',
              'Tochigi' => '09',
              'Gunma' => '10',
              'Saitama' => '11',
              'Chiba' => '12',
              'Tokyo' => '13',
              'Kanagawa' => '14',
              'Niigata' => '15',
              'Toyama' => '16',
              'Ishikawa' => '17',
              'Fukui' => '18',
              'Yamanashi' => '19',
              'Nagano' => '20',
              'Gifu' => '21',
              'Shizuoka' => '22',
              'Aichi' => '23',
              'Mie' => '24',
              'Shiga' => '25',
              'Kyoto' => '26',
              'Osaka' => '27',
              'Hyogo' => '28',
              'Nara' => '29',
              'Wakayama' => '30',
              'Tottori' => '31',
              'Shimane' => '32',
              'Okayama' => '33',
              'Hiroshima' => '34',
              'Yamaguchi' => '35',
              'Tokushima' => '36',
              'Kagawa' => '37',
              'Ehime' => '38',
              'Kochi' => '39',
              'Fukuoka' => '40',
              'Saga' => '41',
              'Nagasaki' => '42',
              'Kumamoto' => '43',
              'Oita' => '44',
              'Miyazaki' => '45',
              'Kagoshima' => '46',
              'Okinawa' => '47',
            ],
            'KN' => [
              'Nevis' => 'N',
              'St. Kitts' => 'K',
            ],
            'KP' => [
              'Kangwon' => '07',
              'Rason' => '13',
              'Ryanggang' => '10',
              'Chagang' => '04',
              'South Pyongan' => '02',
              'North Pyongan' => '03',
              'Pyongyang' => '01',
              'South Hamgyong' => '08',
              'North Hamgyong' => '09',
              'South Hwanghae' => '05',
              'North Hwanghae' => '06',
            ],
            'KR' => [
              'Gangwon-do' => '42',
              'Gyeonggi-do' => '41',
              'Gyeongsangnam-do' => '48',
              'Gyeongsangbuk-do' => '47',
              'Gwangju' => '29',
              'Daegu' => '27',
              'Daejeon' => '30',
              'Busan' => '26',
              'Seoul' => '11',
              'Sejong' => '50',
              'Ulsan' => '31',
              'Incheon' => '28',
              'Jeollanam-do' => '46',
              'Jeollabuk-do' => '45',
              'Jeju-do' => '49',
              'Chungcheongnam-do' => '44',
              'Chungcheongbuk-do' => '43',
            ],
            'MX' => [
              'Ags.' => 'AGU',
              'B.C.' => 'BCN',
              'B.C.S.' => 'BCS',
              'Camp.' => 'CAM',
              'Chis.' => 'CHP',
              'Chih.' => 'CHH',
              'CDMX' => 'CMX',
              'Coah.' => 'COA',
              'Col.' => 'COL',
              'Dgo.' => 'DUR',
              'Méx.' => 'MEX',
              'Gto.' => 'GUA',
              'Gro.' => 'GRO',
              'Hgo.' => 'HID',
              'Jal.' => 'JAL',
              'Mich.' => 'MIC',
              'Mor.' => 'MOR',
              'Nay.' => 'NAY',
              'N.L.' => 'NLE',
              'Oax.' => 'OAX',
              'Pue.' => 'PUE',
              'Qro.' => 'QUE',
              'Q.R.' => 'ROO',
              'S.L.P.' => 'SLP',
              'Sin.' => 'SIN',
              'Son.' => 'SON',
              'Tab.' => 'TAB',
              'Tamps.' => 'TAM',
              'Tlax.' => 'TLA',
              'Ver.' => 'VER',
              'Yuc.' => 'YUC',
              'Zac.' => 'ZAC',
            ],
            'MY' => [
              'Johor' => '01',
              'Kedah' => '02',
              'Kelantan' => '03',
              'Kuala Lumpur' => '14',
              'Labuan' => '15',
              'Melaka' => '04',
              'Negeri Sembilan' => '05',
              'Pahang' => '06',
              'Perak' => '08',
              'Perlis' => '09',
              'Pulau Pinang' => '07',
              'Putrajaya' => '16',
              'Sabah' => '12',
              'Sarawak' => '13',
              'Selangor' => '10',
              'Terengganu' => '11',
            ],
            'MZ' => [
              'Cabo Delgado' => 'P',
              'Cidade de Maputo' => 'MPM',
              'Gaza' => 'G',
              'Inhambane' => 'I',
              'Manica' => 'B',
              'Maputo' => 'L',
              'Nampula' => 'N',
              'Niassa' => 'A',
              'Sofala' => 'S',
              'Tete' => 'T',
              'Zambezia' => 'Q',
            ],
            'NG' => [
              'Abia' => 'AB',
              'Adamawa' => 'AD',
              'Akwa Ibom' => 'AK',
              'Anambra' => 'AN',
              'Bauchi' => 'BA',
              'Bayelsa' => 'BY',
              'Benue' => 'BE',
              'Borno' => 'BO',
              'Cross River' => 'CR',
              'Delta' => 'DE',
              'Ebonyi' => 'EB',
              'Edo' => 'ED',
              'Ekiti' => 'EK',
              'Enugu' => 'EN',
              'Federal Capital Territory' => 'FC',
              'Gombe' => 'GO',
              'Imo' => 'IM',
              'Jigawa' => 'JI',
              'Kaduna' => 'KD',
              'Kano' => 'KN',
              'Katsina' => 'KT',
              'Kebbi' => 'KE',
              'Kogi' => 'KO',
              'Kwara' => 'KW',
              'Lagos' => 'LA',
              'Nasarawa' => 'NA',
              'Niger' => 'NI',
              'Ogun State' => 'OG',
              'Ondo' => 'ON',
              'Osun' => 'OS',
              'Oyo' => 'OY',
              'Plateau' => 'PL',
              'Rivers' => 'RI',
              'Sokoto' => 'SO',
              'Taraba' => 'TA',
              'Yobe' => 'YO',
              'Zamfara' => 'ZA',
            ],
            'NI' => [
              'Boaco' => 'BO',
              'Carazo' => 'CA',
              'Chinandega' => 'CI',
              'Chontales' => 'CO',
              'Estelí' => 'ES',
              'Granada' => 'GR',
              'Jinotega' => 'JI',
              'León' => 'LE',
              'Madriz' => 'MD',
              'Managua' => 'MN',
              'Masaya' => 'MS',
              'Matagalpa' => 'MT',
              'Nueva Segovia' => 'NS',
              'Región Autónoma de la Costa Caribe Norte' => 'AN',
              'Región Autónoma de la Costa Caribe Sur' => 'AS',
              'Río San Juan' => 'SJ',
              'Rivas' => 'RI',
            ],
            'NR' => [
              'Aiwo District' => '01',
              'Anabar District' => '02',
              'Anetan District' => '03',
              'Anibare District' => '04',
              'Baiti District' => '05',
              'Boe District' => '06',
              'Buada District' => '07',
              'Denigomodu District' => '08',
              'Ewa District' => '09',
              'Ijuw District' => '10',
              'Meneng District' => '11',
              'Nibok District' => '12',
              'Uaboe District' => '13',
              'Yaren District' => '14',
            ],
            'PE' => [
              'Amazonas' => 'AMA',
              'Áncash' => 'ANC',
              'Apurímac' => 'APU',
              'Arequipa' => 'ARE',
              'Ayacucho' => 'AYA',
              'Cajamarca' => 'CAJ',
              'Callao' => 'CAL',
              'Cuzco' => 'CUS',
              'Gobierno Regional de Lima' => 'LIM',
              'Huancavelica' => 'HUV',
              'Huánuco' => 'HUC',
              'Ica' => 'ICA',
              'Junín' => 'JUN',
              'La Libertad' => 'LAL',
              'Lambayeque' => 'LAM',
              'Loreto' => 'LOR',
              'Madre de Dios' => 'MDD',
              'Moquegua' => 'MOQ',
              'Municipalidad Metropolitana de Lima' => 'LMA',
              'Pasco' => 'PAS',
              'Piura' => 'PIU',
              'Puno' => 'PUN',
              'San Martín' => 'SAM',
              'Tacna' => 'TAC',
              'Tumbes' => 'TUM',
              'Ucayali' => 'UCA',
            ],
            'PH' => [
              'Abra' => 'ABR',
              'Agusan del Norte' => 'AGN',
              'Agusan del Sur' => 'AGS',
              'Aklan' => 'AKL',
              'Albay' => 'ALB',
              'Antique' => 'ANT',
              'Apayao' => 'APA',
              'Aurora' => 'AUR',
              'Basilan' => 'BAS',
              'Bataan' => 'BAN',
              'Batanes' => 'BTN',
              'Batangas' => 'BTG',
              'Benguet' => 'BEN',
              'Biliran' => 'BIL',
              'Bohol' => 'BOH',
              'Bukidnon' => 'BUK',
              'Bulacan' => 'BUL',
              'Cagayan' => 'CAG',
              'Camarines Norte' => 'CAN',
              'Camarines Sur' => 'CAS',
              'Camiguin' => 'CAM',
              'Capiz' => 'CAP',
              'Catanduanes' => 'CAT',
              'Cavite' => 'CAV',
              'Cebu' => 'CEB',
              'Cotabato' => 'NCO',
              'Compostela Valley' => 'COM',
              'Davao del Norte' => 'DAV',
              'Davao del Sur' => 'DAS',
              'Davao Occidental' => 'DVO',
              'Davao Oriental' => 'DAO',
              'Dinagat Islands' => 'DIN',
              'Eastern Samar' => 'EAS',
              'Guimaras' => 'GUI',
              'Ifugao' => 'IFU',
              'Ilocos Norte' => 'ILN',
              'Ilocos Sur' => 'ILS',
              'Iloilo' => 'ILI',
              'Isabela' => 'ISA',
              'Kalinga' => 'KAL',
              'La Union' => 'LUN',
              'Laguna' => 'LAG',
              'Lanao del Norte' => 'LAN',
              'Lanao del Sur' => 'LAS',
              'Leyte' => 'LEY',
              'Maguindanao' => 'MAG',
              'Marinduque' => 'MAD',
              'Masbate' => 'MAS',
              'Metro Manila' => '00',
              'Misamis Occidental' => 'MSC',
              'Misamis Oriental' => 'MSR',
              'Mountain Province' => 'MOU',
              'Negros Occidental' => 'NEC',
              'Negros Oriental' => 'NER',
              'Northern Samar' => 'NSA',
              'Nueva Ecija' => 'NUE',
              'Nueva Vizcaya' => 'NUV',
              'Mindoro Occidental' => 'MDC',
              'Mindoro Oriental' => 'MDR',
              'Palawan' => 'PLW',
              'Pampanga' => 'PAM',
              'Pangasinan' => 'PAN',
              'Quezon Province' => 'QUE',
              'Quirino' => 'QUI',
              'Rizal' => 'RIZ',
              'Romblon' => 'ROM',
              'Samar' => 'WSA',
              'Sarangani' => 'SAR',
              'Siquijor' => 'SIG',
              'Sorsogon' => 'SOR',
              'South Cotabato' => 'SCO',
              'Southern Leyte' => 'SLE',
              'Sultan Kudarat' => 'SUK',
              'Sulu' => 'SLU',
              'Surigao del Norte' => 'SUN',
              'Surigao del Sur' => 'SUR',
              'Tarlac' => 'TAR',
              'Tawi-Tawi' => 'TAW',
              'Zambales' => 'ZMB',
              'Zamboanga del Norte' => 'ZAN',
              'Zamboanga del Sur' => 'ZAS',
              'Zamboanga Sibuguey' => 'ZSI',
            ],
            'RU' => [
              'Altayskiy kray' => 'ALT',
              'Amurskaya oblast\'' => 'AMU',
              'Arkhangelskaya oblast\'' => 'ARK',
              'Astrakhanskaya oblast\'' => 'AST',
              'Belgorodskaya oblast\'' => 'BEL',
              'Bryanskaya oblast\'' => 'BRY',
              'Vladimirskaya oblast\'' => 'VLA',
              'Volgogradskaya oblast\'' => 'VGG',
              'Vologodskaya oblast\'' => 'VLG',
              'Voronezhskaya oblast\'' => 'VOR',
              'Evreyskaya avtonomnaya oblast\'' => 'YEV',
              'Zabaykalskiy kray' => 'ZAB',
              'Ivanovskaya oblast\'' => 'IVA',
              'Irkutskaya oblast\'' => 'IRK',
              'Kabardino-Balkarskaya Republits' => 'KB',
              'Kaliningradskaya oblast\'' => 'KGD',
              'Kaluzhskaya oblast\'' => 'KLU',
              'Kamchatskiy kray' => 'KAM',
              'Karachaevo-Cherkesskaya Republits' => 'KC',
              'Kemerovskaya oblast\'' => 'KEM',
              'Kirovskaya oblast\'' => 'KIR',
              'Kostromskaya oblast\'' => 'KOS',
              'Krasnodarskiy kray' => 'KDA',
              'Krasnoyarskiy kray' => 'KYA',
              'Kurganskaya oblast\'' => 'KGN',
              'Kurskaya oblast\'' => 'KRS',
              'Leningradskaya oblast\'' => 'LEN',
              'Lipetskaya oblast\'' => 'LIP',
              'Magadanskaya oblast\'' => 'MAG',
              'Moskva' => 'MOW',
              'Moskovskaya oblast\'' => 'MOS',
              'Murmanskaya oblast\'' => 'MUR',
              'Nenetskiy' => 'NEN',
              'Nizhegorodskaya oblast\'' => 'NIZ',
              'Novgorodskaya oblast\'' => 'NGR',
              'Novosibirskaya oblast\'' => 'NVS',
              'Omskaya oblast\'' => 'OMS',
              'Orenburgskaya oblast\'' => 'ORE',
              'Orlovskaya oblast\'' => 'ORL',
              'Penzenskaya oblast\'' => 'PNZ',
              'Permskiy kray' => 'PER',
              'Primorskiy kray' => 'PRI',
              'Pskovskaya oblast\'' => 'PSK',
              'Respublika Adygeya' => 'AD',
              'Altay Republits' => 'AL',
              'Bashkortostan Republits' => 'BA',
              'Buryatiya Republits' => 'BU',
              'Dagestan Republits' => 'DA',
              'Ingushetiya Republits' => 'IN',
              'Respublika Kalmykiya' => 'KL',
              'Kareliya Republits' => 'KR',
              'Komi Republits' => 'KO',
              'Respublika Mariy El' => 'ME',
              'Respublika Mordoviya' => 'MO',
              'Sakha (Yakutiya) Republits' => 'SA',
              'Respublika Severnaya Osetiya-Alaniya' => 'SE',
              'Respublika Tatarstan' => 'TA',
              'Tyva Republits' => 'TY',
              'Udmurtskaja Respublika' => 'UD',
              'Khakasiya Republits' => 'KK',
              'Rostovskaya oblast\'' => 'ROS',
              'Ryazanskaya oblast\'' => 'RYA',
              'Samarskaya oblast\'' => 'SAM',
              'Sankt-Peterburg' => 'SPE',
              'Saratovskaya oblast\'' => 'SAR',
              'Sakhalinskaya oblast\'' => 'SAK',
              'Sverdlovskaya oblast\'' => 'SVE',
              'Smolenskaya oblast\'' => 'SMO',
              'Stavropolskiy kray' => 'STA',
              'Tambovskaya oblast\'' => 'TAM',
              'Tverskaya oblast\'' => 'TVE',
              'Tomskaya oblast\'' => 'TOM',
              'Tulskaya oblast\'' => 'TUL',
              'Tyumenskaya oblast\'' => 'TYU',
              'Ulyanovskaya oblast\'' => 'ULY',
              'Khabarovskiy kray' => 'KHA',
              'Khanty-Mansiyskiy avtonomnyy okrug' => 'KHM',
              'Chelyabinskaya oblast\'' => 'CHE',
              'Chechenskaya Republits' => 'CE',
              'Chuvashia' => 'CU',
              'Chukotskiy' => 'CHU',
              'Yamalo-Nenetskiy' => 'YAN',
              'Yaroslavskaya oblast\'' => 'YAR',
            ],
            'SR' => [
              'Brokopondo' => 'BR',
              'Commewijne' => 'CM',
              'Coronie' => 'CR',
              'Marowijne' => 'MA',
              'Nickerie' => 'NI',
              'Para' => 'PR',
              'Paramaribo' => 'PM',
              'Saramacca' => 'SA',
              'Sipaliwini' => 'SI',
              'Wanica' => 'WA',
            ],
            'SV' => [
              'Ahuachapan' => 'AH',
              'Cabanas' => 'CA',
              'Calatenango' => 'CH',
              'Cuscatlan' => 'CU',
              'La Libertad' => 'LI',
              'La Paz' => 'PA',
              'La Union' => 'UN',
              'Morazan' => 'MO',
              'San Miguel' => 'SM',
              'San Salvador' => 'SS',
              'San Vicente' => 'SV',
              'Santa Ana' => 'SA',
              'Sonsonate' => 'SO',
              'Usulutan' => 'US',
            ],
            'TH' => [
              'Krabi' => '81',
              'Bangkok' => '10',
              'Kanchanaburi' => '71',
              'Kalasin' => '46',
              'Kamphaeng Phet' => '62',
              'Khon Kaen' => '40',
              'Bueng Kan' => '38',
              'Chanthaburi' => '22',
              'Chachoengsao' => '24',
              'Chon Buri' => '20',
              'Chai Nat' => '18',
              'Chaiyaphum' => '36',
              'Chumpon' => '86',
              'Chiang Rai' => '57',
              'Chiang Mai' => '50',
              'Trang' => '92',
              'Trat' => '23',
              'Tak' => '63',
              'Nakhon Nayok' => '26',
              'Nakhon Pathom' => '73',
              'Nakhon Phanom' => '48',
              'Nakhon Ratchasima' => '30',
              'Nakhon Si Thammarat' => '80',
              'Nakhon Sawan' => '60',
              'Nonthaburi' => '12',
              'Narathiwat' => '96',
              'Nan' => '55',
              'Buri Ram' => '31',
              'Pathum Thani' => '13',
              'Prachuap Khiri Khan' => '77',
              'Prachin Buri' => '25',
              'Pattani' => '94',
              'Phra Nakhon Si Ayutthaya' => '14',
              'Phayao' => '56',
              'Phang Nga' => '82',
              'Phattalung' => '93',
              'Phichit' => '66',
              'Phitsanulok' => '65',
              'Phetchaburi' => '76',
              'Phetchabun' => '67',
              'Phrae' => '54',
              'Phuket' => '83',
              'Maha Sarakham' => '44',
              'Mukdahan' => '49',
              'Mae Hong Son' => '58',
              'Yasothon' => '35',
              'Yala' => '95',
              'Roi Et' => '45',
              'Ranong' => '85',
              'Rayong' => '21',
              'Ratchaburi' => '70',
              'Lop Buri' => '16',
              'Lampang' => '52',
              'Lamphun' => '51',
              'Loei' => '42',
              'Si Sa Ket' => '33',
              'Sakon Nakhon' => '47',
              'Songkhla' => '90',
              'Satun' => '91',
              'Samut Prakan' => '11',
              'Samut Songkhram' => '75',
              'Samut Sakhon' => '74',
              'Sa Kaeo' => '27',
              'Saraburi' => '19',
              'Sing Buri' => '17',
              'Sukhothai' => '64',
              'Suphanburi' => '72',
              'Surat Thani' => '84',
              'Surin' => '32',
              'Nong Khai' => '43',
              'Nong Bua Lam Phu' => '39',
              'Ang Thong' => '15',
              'Amnat Charoen' => '37',
              'Udon Thani' => '41',
              'Uttaradit' => '53',
              'Uthai Thani' => '61',
              'Ubon Ratchathani' => '34',
            ],
            'TR' => [
              'Adana' => '01',
              'Adıyaman' => '02',
              'Afyon' => '03',
              'Ağrı' => '04',
              'Aksaray' => '68',
              'Amasya' => '05',
              'Ankara' => '06',
              'Antalya' => '07',
              'Ardahan' => '75',
              'Artvin' => '08',
              'Aydın' => '09',
              'Balıkesir' => '10',
              'Bartın' => '74',
              'Batman' => '72',
              'Bayburt' => '69',
              'Bilecik' => '11',
              'Bingöl' => '12',
              'Bitlis' => '13',
              'Bolu' => '14',
              'Burdur' => '15',
              'Bursa' => '16',
              'Çanakkale' => '17',
              'Çankırı' => '18',
              'Çorum' => '19',
              'Denizli' => '20',
              'Diyarbakır' => '21',
              'Düzce' => '81',
              'Edirne' => '22',
              'Elazığ' => '23',
              'Erzincan' => '24',
              'Erzurum' => '25',
              'Eskişehir' => '26',
              'Gaziantep' => '27',
              'Giresun' => '28',
              'Gümüşhane' => '29',
              'Hakkari' => '30',
              'Hatay' => '31',
              'Iğdır' => '76',
              'Isparta' => '32',
              'İstanbul' => '34',
              'İzmir' => '35',
              'Kahramanmaraş' => '46',
              'Karabük' => '78',
              'Karaman' => '70',
              'Kars' => '36',
              'Kastamonu' => '37',
              'Kayseri' => '38',
              'Kırıkkale' => '71',
              'Kırklareli' => '39',
              'Kırşehir' => '40',
              'Kilis' => '79',
              'Kocaeli' => '41',
              'Konya' => '42',
              'Kütahya' => '43',
              'Malatya' => '44',
              'Manisa' => '45',
              'Mardin' => '47',
              'Mersin' => '33',
              'Muğla' => '48',
              'Muş' => '49',
              'Nevşehir' => '50',
              'Niğde' => '51',
              'Ordu' => '52',
              'Osmaniye' => '80',
              'Rize' => '53',
              'Sakarya' => '54',
              'Samsun' => '55',
              'Siirt' => '56',
              'Sinop' => '57',
              'Sivas' => '58',
              'Şanlıurfa' => '63',
              'Şırnak' => '73',
              'Tekirdağ' => '59',
              'Tokat' => '60',
              'Trabzon' => '61',
              'Tunceli' => '62',
              'Uşak' => '64',
              'Van' => '65',
              'Yalova' => '77',
              'Yozgat' => '66',
              'Zonguldak' => '67',
            ],
            'TV' => [
              'Funafuti' => 'FUN',
              'Nanumanga' => 'NMG',
              'Nanumea' => 'NMA',
              'Niutao' => 'NIT',
              'Nui' => 'NUI',
              'Nukufetau' => 'NKF',
              'Nukulaelae' => 'NKL',
              'Vaitupu' => 'VAI',
            ],
            'TW' => [
              'Keelung City' => 'KEE',
              'Taipei City' => 'TPE',
              'New Taipei City' => 'NWT',
              'Taoyuan City' => 'TAO',
              'Hsinchu City' => 'HSZ',
              'Hsinchu County' => 'HSQ',
              'Miaoli County' => 'MIA',
              'Taichung City' => 'TXG',
              'Changhua County' => 'CHA',
              'Nantou County' => 'NAN',
              'Yunlin County' => 'YUN',
              'Chiayi City' => 'CYI',
              'Chiayi County' => 'CYQ',
              'Tainan City' => 'TNN',
              'Kaohsiung City' => 'KHH',
              'Pingtung County' => 'PIF',
              'Taitung County' => 'TTT',
              'Hualien County' => 'HUA',
              'Yilan County' => 'ILA',
              'Penghu County' => 'PEN',
              'Kinmen County' => 'KIN',
              'Lienchiang County' => 'LIE',
            ],
            'UA' => [
              'Crimea' => '43',
              'Vinnyts\'ka oblast' => '05',
              'Volyns\'ka oblast' => '07',
              'Dnipropetrovsk oblast' => '12',
              'Donetsk oblast' => '14',
              'Zhytomyrs\'ka oblast' => '18',
              'Zakarpats\'ka oblast' => '21',
              'Zaporiz\'ka oblast' => '23',
              'Ivano-Frankivs\'ka oblast' => '26',
              'Kyiv city' => '30',
              'Kiev oblast' => '32',
              'Kirovohrads\'ka oblast' => '35',
              'Luhans\'ka oblast' => '09',
              'Lviv oblast' => '46',
              'Mykolaivs\'ka oblast' => '48',
              'Odessa oblast' => '51',
              'Poltavs\'ka oblast' => '53',
              'Rivnens\'ka oblast' => '56',
              'Sevastopol\' city' => '40',
              'Sums\'ka oblast' => '59',
              'Ternopil\'s\'ka oblast' => '61',
              'Kharkiv oblast' => '63',
              'Khersons\'ka oblast' => '65',
              'Khmel\'nyts\'ka oblast' => '68',
              'Cherkas\'ka oblast' => '71',
              'Chernivets\'ka oblast' => '77',
              'Chernihivs\'ka oblast' => '74',
            ],
            'UY' => [
              'Artigas' => 'AR',
              'Canelones' => 'CA',
              'Cerro Largo' => 'CL',
              'Colonia' => 'CO',
              'Durazno' => 'DU',
              'Flores' => 'FS',
              'Florida' => 'FD',
              'Lavalleja' => 'LA',
              'Maldonado' => 'MA',
              'Montevideo' => 'MO',
              'Paysandú' => 'PA',
              'Río Negro' => 'RN',
              'Rivera' => 'RV',
              'Rocha' => 'RO',
              'Salto' => 'SA',
              'San José' => 'SJ',
              'Soriano' => 'SO',
              'Tacuarembó' => 'TA',
              'Treinta y Tres' => 'TT',
            ],
            'VE' => [
              'Amazonas' => 'Z',
              'Anzoátegui' => 'B',
              'Apure' => 'C',
              'Aragua' => 'D',
              'Barinas' => 'E',
              'Bolívar' => 'F',
              'Carabobo' => 'G',
              'Cojedes' => 'H',
              'Delta Amacuro' => 'Y',
              'Dependencias Federales' => 'W',
              'Distrito Federal' => 'A',
              'Falcón' => 'I',
              'Guárico' => 'J',
              'Lara' => 'K',
              'Mérida' => 'L',
              'Miranda' => 'M',
              'Monagas' => 'N',
              'Nueva Esparta' => 'O',
              'Portuguesa' => 'P',
              'Sucre' => 'R',
              'Táchira' => 'S',
              'Trujillo' => 'T',
              'Vargas' => 'X',
              'Yaracuy' => 'U',
              'Zulia' => 'V',
            ],
            'VN' => [
              'An Giang Province' => '44',
              'Ba Ria-Vung Tau Province' => '43',
              'Bac Lieu Province' => '55',
              'Bac Giang Province' => '54',
              'Bac Kan Province' => '53',
              'Bac Ninh Province' => '56',
              'Ben Tre Province' => '50',
              'Binh Duong Province' => '57',
              'Binh Dinh Province' => '31',
              'Binh Phuoc Province' => '58',
              'Binh Thuan Province' => '40',
              'Ca Mau Province' => '59',
              'Cao Bang Province' => '04',
              'Can Tho City' => 'CT',
              'Da Nang City' => 'DN',
              'Dak Lak Province' => '33',
              'Dak Nong Province' => '72',
              'Dien Bien Province' => '71',
              'Dong Nai Province' => '39',
              'Dong Thap Province' => '45',
              'Gia Lai Province' => '30',
              'Ha Giang Province' => '03',
              'Ha Nam Province' => '63',
              'Hanoi City' => 'HN',
              'Ha Tinh Province' => '23',
              'Hai Duong Province' => '61',
              'Haiphong City' => 'HP',
              'Hau Giang Province' => '73',
              'Hoa Binh Province' => '14',
              'Hung Yen Province' => '66',
              'Khanh Hoa Province' => '34',
              'Kien Giang Province' => '47',
              'Kon Tum Province' => '28',
              'Lai Chau Province' => '01',
              'Lang Song Province' => '09',
              'Lao Cai Province' => '02',
              'Lam Dong Province' => '35',
              'Long An Province' => '41',
              'Nam Dinh Province' => '67',
              'Nghe An Province' => '22',
              'Ninh Binh Province' => '18',
              'Ninh Thuan Province' => '36',
              'Phu Tho Province' => '68',
              'Phu Yen Province' => '32',
              'Quang Binh Province' => '24',
              'Quang Nam Province' => '27',
              'Quang Ngai Province' => '29',
              'Quang Ninh Province' => '13',
              'Quang Tri Province' => '25',
              'Soc Trang Province' => '52',
              'Son La Province' => '05',
              'Tay Ninh Province' => '37',
              'Thai Binh Province' => '20',
              'Thai Nguyen Province' => '69',
              'Thanh Hoa Province' => '21',
              'Ho Chi Minh City' => 'SG',
              'Thua Thien-Hue Province' => '26',
              'Tien Giang Province' => '46',
              'Tra Vinh Province' => '51',
              'Tuyen Quang Province' => '07',
              'Vinh Long Province' => '49',
              'Vinh Phuc Province' => '70',
              'Yen Bai Province' => '06',
            ],
          ];
    }
}
